'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module for Textcreation                                       #
#                                                                               #
#################################################################################
#	C O N T E N T    					                                     	#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 nix                                                              #
#        02.03 update manager                                                   #
#        02.04 calculate scale factor                                           #
#        02.05 interpolate font outlines                                        #
#        02.06 calculate contour order                                          #
#        02.07 calculate cutpath offset                                         #
#        02.08 scale fonts                                                      #
#        02.09 rotate Text                                                      #
#        02.10 mirror Text                                                      #
#        02.11 startcodes                                                       #  
#        02.12 g-code generation for module                                     #
#        02.13 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import vecFont
import uni
from copy import copy, deepcopy
import time
if sys.platform == "win32":
    from PyQt5.QtCore import QCoreApplication #Win
elif sys.platform == "linux2":
    from PyQt4.QtCore import QCoreApplication #Debian


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "TrueType-Text"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "TrueTypeText.png"

#---explaining text for helpbutton in lonnox cut---
info = "TextInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","F","T","N","N","N","N","N","N","O","N","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","Fontfile","Text",
             "X","Y","Size in mm", "Letterdistance", "Rotation", 
             "Bow radius", "Cutpath", "Depth","Cutdepth", "Feed",
             "Interpolation-Quality")
             
#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror"),
             "Cutpath":          ("Center","Inside","Outside")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 
        
        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        file          = jParams[jIndex][3] 
        text          = jParams[jIndex][4]
        x             = jParams[jIndex][5] # + x0  see gcode generation
        y             = jParams[jIndex][6] # + y0  see gcode generation
        size          = jParams[jIndex][7]                      
        dis           = jParams[jIndex][8]
        rotation      = jParams[jIndex][9]  
        bow           = jParams[jIndex][10]  
        cutpath       = jParams[jIndex][11]
        depth         = math.fabs( jParams[jIndex][12] )
        cutdepth      = math.fabs( jParams[jIndex][13] )
        feed          = jParams[jIndex][14]                             
        fPlunge       = feed * uni.settings[6] / 100         
        div           = int( jParams[jIndex][15] )                              
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
  
        #---set preview values---
        if preview: depth=1; cutdepth=1        
         
        #---clamp Interpolation division---
        if div < 1: div = 1
        
        #---escape if essetial values are missing---
        if (not depth or not cutdepth or not size or 
           (file=="") or (text=="") or (file[-4:] != ".ttf") ): 
            return code

        #########################################################################
        #    02.03 update manager                                               #
        #########################################################################
        
        #---save with modulename/cachename---        
        cn = jNames[jIndex].replace(" ","_") 
        
        #---check if update is needed---
        try: 
            if (uni.cache[cn][0] != cr or
                uni.cache[cn][1] != file or
                uni.cache[cn][2] != text or
                uni.cache[cn][3] != cutpath or
                uni.cache[cn][4] != div or
                uni.cache[cn][7] != size or
                uni.cache[cn][8] != rotation):
                refresh = 1
            else: refresh = 0                 
        except: uni.cache[cn] = [None,None,None,None,None,None,None,None,None]; refresh = 1 

        #---update font data---
        if refresh:
            #---read font data---
            if os.path.exists(file):
                fontData = vecFont.read( self, text, file )
                tDetail = fontData[0]
                glyphs = fontData[1]
            else: return "(Schriftdatei nicht gefunden: " + file + ")"     
        else: 
            #---reload font data---
            tDetail = uni.cache[cn][5] 
            glyphs = deepcopy(uni.cache[cn][6]) 
        
        #---save/rewrite relevant values---        
        uni.cache[cn][0] = cr
        uni.cache[cn][1] = file
        uni.cache[cn][2] = text
        uni.cache[cn][3] = cutpath
        uni.cache[cn][4] = div
        uni.cache[cn][5] = tDetail
        uni.cache[cn][7] = size
        uni.cache[cn][8] = rotation
        
        
        #########################################################################
        #    02.04 calculate scale factor                                       #
        #########################################################################

        #---calculate scale factor for fontsize---
        scale = size / (tDetail["yMax"] - tDetail["yMin"]) 
        
        
        #########################################################################
        #    02.05 interpolate font outlines                                    #
        #########################################################################

        #---interpolate coordinates---        
        if refresh: 
            pAmount = 0
            glyphsI = []
            for glyph in glyphs: 
                glyphI = {"contours":[],"width":glyph["width"]}
                for cont in glyph["contours"]:        
                    contI = []
                    p0 = [0,0,0]  
                    p1 = [0,0,0]  
                    for p2 in cont:
                        #---if previous point was a control point use quad spline---
                        if p1[2] and not p2[2]:
                            for i in range(div):
                                t = (1/div) * (i+1)
                                xP = ((1-t)**2 * p0[0]) + (2 * (1-t) * t * p1[0]) + (t**2 * p2[0])
                                yP = ((1-t)**2 * p0[1]) + (2 * (1-t) * t * p1[1]) + (t**2 * p2[1])
                                contI.append( [xP,yP] )
                                pAmount += 1
                        #---if previous and current point are no c. points use straight line---
                        elif not p1[2] and not p2[2]: 
                            contI.append(p2[:2])
                            pAmount += 1
                        #---save previous point---
                        p0 = list(p1)
                        p1 = list(p2)
                    glyphI["contours"].append(list(contI)) 
                glyphsI.append(glyphI) 
            glyphs = list(glyphsI)            


        #########################################################################
        #    02.06 calculate contour order                                      #
        #########################################################################
           
        #---search highest distance of every contour---
        if refresh: 
            for index,glyph in enumerate(glyphs):
                order = []
                for cont in glyph["contours"]:
                    dmax = 0
                    for p in cont:
                        d = math.fabs( p[0]**2 + p[1]**2 )
                        if d > dmax: dmax = d
                    order.append( dmax )                
                glyph2 = {"contours":[]}
                for i in order:
                    j = order.index(max(order))
                    glyph2["contours"].insert( 0,deepcopy(glyph["contours"][j]) )            
                    order[j] = 0
                glyph["contours"] = glyph2["contours"]
                
            
        #########################################################################
        #    02.07 calculate cutpath offset                                     #
        #########################################################################

        #---calculate cutpath---
        if cutpath != "Center" and refresh:
            progress = 0 
            uni.key = ""
            #---scale cutter radius---
            crS = cr/scale
            #---loop for every char---
            glyphs2 = []
            for glyph in glyphs: 
                #---loop for every contour---
                glyph2 = {"contours":[],"width":glyph["width"]}
                for cont in glyph["contours"]:        
                    #---create new dictonary (not writeable)---
                    cont2 = []
                    contL = len(cont)
                    cont.insert(0, list(cont[-2]) )
                    cont.append( list(cont[2]) )
                    #---loop for every new point---
                    for i in range(contL):
                        #---show progress in mcad---
                        progress += 1
                        uni.moduleStatus = L["Calculating textpath (ESC)..."] + "{:.0f}%".format(100/pAmount*progress)  
                        QCoreApplication.processEvents() 
                        if uni.key == "ESC": break
                        #---calculate new point with cutter radius offset--- 
                        pC = gcode.pContour( self, cont[i][:2], cont[i+1][:2], cont[i+2][:2], crS )
                        if cutpath == "Inside":  
                            xCont2 = pC["isecRX"] 
                            yCont2 = pC["isecRY"]
                        else: 
                            xCont2 = pC["isecLX"]
                            yCont2 = pC["isecLY"]
                        #---add new point---
                        cont2.append( [xCont2,yCont2] )
                        #---check the distance from new point to countour---
                        for j in range(contL-1):
                            line = (cont[j][0],cont[j][1],cont[j+1][0],cont[j+1][1])
                            lPData = gcode.distanceLP( self, line, (xCont2,yCont2) ) 
                            if lPData[1] and ((crS*0.98) > lPData[0]): del cont2[-1]; break
                    #---if no contour point stay over after distance check then skip---  
                    if len(cont2) > 0:
                        #---close contour if it isn't---          
                        if cont2[0] != cont2[-1]: cont2.append(list(cont2[0]))
                        #---save new points--- 
                        glyph2["contours"].append( list(cont2) )
                glyphs2.append( glyph2 )
            glyphs = list(glyphs2)            
            uni.moduleStatus = ""
           
        #---save/rewrite font data for update manager---
        if refresh: uni.cache[cn][6] = deepcopy(glyphs)
        
            
        #########################################################################
        #    02.08 scale fonts                                                  #
        #########################################################################

        #---calculate scale factor for fontsize---
        y -= tDetail["yMin"] * scale

        #---scale coordinates---
        for glyph in glyphs: 
            for cont in glyph["contours"]:        
                for p in cont:
                    p[0] *= scale
                    p[1] *= scale
            glyph["sWidth"] = glyph["width"] * scale                              
            
            
        #########################################################################
        #    02.09 rotate text                                                  #
        #########################################################################

        #---check bow radius---
        for glyph in glyphs:
            if ((glyph["sWidth"]+dis) >= math.fabs(bow*2)) and (bow != 0): 
                uni.moduleStatus = L["Bow radius is to small/ font is to big!"]
                bow = 0
                break;
            
        #---rotate x/y values of the letters---
        xBase = x; yBase = y; aBow = 0
        for glyph in glyphs:
            #---BOW ROTATION: calculate letter rotaion with bow radius---
            if bow >= 0: aBow -= gcode.fontBow( self, math.fabs(bow), glyph["sWidth"]+dis ) 
            else: aBow += gcode.fontBow( self, math.fabs(bow), glyph["sWidth"]+dis ) 
            for cont in glyph["contours"]:        
                for p in cont:
                    
                    #---BOW ROTATION: calculate distance and angle of original point---
                    rP = gcode.distancePP( self, (0,0), (p[0],p[1]) )
                    aP = gcode.pointAngle360( self, 0,0, p[0],p[1], metric=1 ) + aBow
                    while aP < 0 : aP += 360 
                    while aP >= 360: aP -= 360         
                    #---BOW ROTATION: rotate points along the bow---
                    p[0],p[1] = gcode.circlePoint( self, xBase, yBase, aP, rP ) 
                    
                    #---TEXT ROTATION: calculate distance and angle of original point---
                    rP = gcode.distancePP( self, (0,0), (p[0],p[1]) )
                    aP = rotation + gcode.pointAngle360( self, 0,0, p[0],p[1], metric=1 )
                    while aP < 0 : aP += 360 
                    while aP >= 360: aP -= 360         
                    #---TEXT ROTATION: rotate points around x=0/y=0---
                    p[0],p[1] = gcode.circlePoint( self, 0, 0, aP, rP ) 
                    
            #---calculate next letters base point---
            aBase = aBow
            while aBase < 0 : aBase += 360 
            while aBase >= 360: aBase -= 360         
            xBase,yBase = gcode.circlePoint( self, xBase, yBase, aBase, glyph["sWidth"]+dis ) 
                    

        #########################################################################
        #    02.010 mirror Text                                                 #
        #########################################################################

        #---mirror x values---
        if orient == "Mirror":
            for glyph in glyphs: 
                for cont in glyph["contours"]:        
                    for p in cont:
                        p[0] = mirrorstops-p[0]

                        
        #########################################################################
        #    02.11 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )

        
        #########################################################################
        #    02.12 g-code generation for module                                 #
        #########################################################################

        #---loop for every char---
        for i,glyph in enumerate(glyphs): 
            
            #---loop for every contour---
            for cont in glyph["contours"]:
                #---move to start position of contour---
                code += "G0 Z{:.3f}\n".format( thick+zSave )
                code += "G0 X{:.3f} Y{:.3f}\n".format( x0+cont[0][0], y0+cont[0][1] )
                
                #---loop until final depth was reached---
                currentdepth = 0.00
                while currentdepth < depth:

                    #---set next depth level or final depth---
                    currentdepth += cutdepth
                    if currentdepth > depth: currentdepth = depth

                    #---move to depth---
                    code += "G1 Z{:.3f} F{:.3f}\n".format( thick-currentdepth, fPlunge )
                    code += "F{:.3f}\n".format( feed )

                    #---loop for every point in contour---
                    for p in cont:
                        code += "G1 X{:.3f} Y{:.3f}\n".format( x0+p[0], y0+p[1] )
                        
                #---move z up---
                code += "G0 Z" + str( thick + zSave ) + "\n\n"
               

                  
        #########################################################################
        #    02.13 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )

        return code 
   

   
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)
   
