'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Spline Group                                          #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 change values by options                                         #
#        02.03 calculate cutpath and open/close options                         #
#        02.04 calculate tangential roundings                                   #
#        02.05 create list for forth and back direction                         #
#        02.06 add colsed contour position                                      #
#        02.07 startcodes                                                       #
#        02.08 g-code generation for module                                     #
#        02.09 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni
from geomdl import BSpline, NURBS, fitting
from geomdl import utilities as geoUtil
#The Directory "/LonnoxCUT/geomdl" is mandatory for the linux Version

#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Spline-Group"
groupEnd   = "Spline-End"
 
#---explain picture of the module--- 
picture    = "SplineG.png"

#---explaining text for helpbutton in lonnox cut---
info = "SplineGInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","O","O","O","O","N","O","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","Spline Type",
             "Point Type","Quality","Contour Type","Rotation","Cutpath","Depth",
             "Cutdepth","Feed")

#---option presets for listed parameters---
options   = {"Orientation":        ("Normal","Mirror"),
             "Spline Type":        ("Quadratic B-Spline","Cubic B-Spline"), 
             "Point Type":         ("Control Points","Fit Points"),
             "Contour Type":       ("Open","Close"),
             "Quality":            ("Low","Medium","High"),             
             "Cutpath":            ("Center","Left","Right") }



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 

        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
       
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        splType       = jParams[jIndex][3]
        pntType       = jParams[jIndex][4]
        quality       = jParams[jIndex][5]
        conType       = jParams[jIndex][6]
        rotation      = jParams[jIndex][7]  
        cutpath       = jParams[jIndex][8] 
        depth         = math.fabs( jParams[jIndex][9] )
        cutdepth      = math.fabs( jParams[jIndex][10] )
        f             = jParams[jIndex][11]   
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
        
        #---search and collect spline points---
        xyw = []  #PointX,PointY,Weight
        pNames = [] #PointNames
        i = jIndex+1
        while i < len(jNames): 
            p = jParams[i]
            #---add point---
            if jNames[i][:-4] == "Spline-Point": 
                newP = [x0+p[0]+p[2], y0+p[1]+p[3], p[4]]
                #---correct weigt if it is zero---
                if newP[2] == 0: newP[2] = 1.0
                #---add point if not already exist---
                if not newP in xyw:
                    xyw.append( newP )
                    pNames.append( jNames[i][-3:] )
            #---cancel on first "non-spline-job"---
            else: break 
            i += 1
            
        #---set spline degree---
        if splType == "Cubic B-Spline": degree = 3
        else: degree = 2 
        
        #---set preview values---
        if preview: depth=2; cutdepth=1

        #---escape if essetial values are missing---
        if not depth or not cutdepth or len(xyw) <= degree: return code


        #########################################################################
        #    02.02 change values by options                                     #
        #########################################################################
        
        #---rotate coords---
        if rotation != 0:
            rad = math.radians(rotation)
            for i,(x,y,w) in enumerate(xyw):
                x -= x0
                y -= y0
                xr = math.cos(rad) * x - math.sin(rad) * y
                yr = math.sin(rad) * x + math.cos(rad) * y
                xyw[i] = [x0+xr,y0+yr,w]
       
        #---mirror points---
        if orient == "Mirror":
            for i in range(len(xyw)): 
                xyw[i][0] = mirrorstops - xyw[i][0]

        #---build point names after mirrorring--- 
        for i,(x,y,w) in enumerate(xyw):
            pNames[i] = "(.{}{} X{} Y{})\n".format(L["Spline-Point"][:1],pNames[i],x,y)
            
        #---set smoothing---
        div = len(xyw) * 25
        if   quality == "Medium": div = len(xyw) * 100
        elif quality == "High"  : div = len(xyw) * 300
        
        
        #########################################################################
        #    02.03 calculate spline interpolation points                        #
        #########################################################################
        #https://github.com/orbingol/geomdl-examples
        #https://nurbs-python.readthedocs.io/en/latest/module_nurbs.html#nurbs-curve
        #https://nurbs-python.readthedocs.io/en/latest/fitting.html
        
        #---build spline by fitpoints-------------------------------------------
        # splines createt by fitpoints can lead to different results because
        # there are different mathematically methods available. geomdb, scipy
        # and Autocad all have different results.
        if pntType == "Fit Points":
            #---build closed spline---
            if conType == "Close":
                #---build open spline with overlength---
                xS,yS = xyw[0][:2]
                xyw = xyw[len(xyw)-degree:len(xyw)] + xyw + xyw[:degree]
                curve = fitting.interpolate_curve([[x,y] for x,y,w in xyw],degree)
                curve.sample_size = div
                xyInterp = curve.evalpts
                
                #---search first last segment---
                dis = [] #[distance,i]
                for i,(x,y) in enumerate(xyInterp):
                    dis.append( [math.sqrt( (xS-x)**2 + (yS-y)**2 ),i] )
                dis = sorted( dis )

                #---use smallest distances with at least div/5 segments between---
                start = dis[0][1]
                i = 1
                while math.fabs( start-dis[i][1] ) < (div/5): i += 1
                end = dis[i][1]
                
                #---cut from fist to last segment and close (cut overlength)---
                xyInterp = xyInterp[min(start,end):max(start,end)]
                xyInterp = xyInterp + xyInterp[:1]
            #---build open spline--- 
            else:
                curve = fitting.interpolate_curve([[x,y] for x,y,w in xyw],degree)
                curve.sample_size = div
                xyInterp = curve.evalpts
            #---scipy variant of fitting (maby better results?)---
            #npX = np.array([x for x,y,w in xyw])
            #npY = np.array([y for x,y,w in xyw])
            #if conType == "Close": close = True
            #else: close = False
            #t, u = splprep([npX, npY], k=degree, per=close, s=0)
            #splX, splY = splev(np.linspace(0,1,div), t)
            #xyInterp = np.vstack((splX, splY)).T.tolist() #Transpose
        #---build spline by control points--------------------------------------
        else:
            #---setup b spline object (called NURBS if weights involved)--- 
            curve = NURBS.Curve() 
            curve.degree = degree 
            
            #---set closed spline parameter---
            if conType == "Close":
                xyw = xyw + xyw[:degree]
                curve.ctrlptsw = xyw
                curve.knotvector = [i for i in range(len(xyw)+degree+1)]
            else:
                curve.ctrlptsw = xyw
                curve.knotvector = geoUtil.generate_knot_vector(degree,len(xyw))
                
            #---calculate spline---
            curve.sample_size = div
            xyInterp = curve.evalpts        

          
        #########################################################################
        #    02.04 calculate cutpath                                            #
        #########################################################################
        
        if cutpath != "Center" and cr > 0:

            #---add first and last element for first last intersection calc---
            xyInterp.insert( 0, xyInterp[-1].copy() )        
            xyInterp.append( xyInterp[1].copy() )

            #---shift the lines of interpolated spline---
            xyCPath = []
            for i in range(len(xyInterp)-2):
                #---get previous, current and next point---
                pa=xyInterp[i]; pb=xyInterp[i+1]; pc=xyInterp[i+2]
                #---calculate parallel contour points---  
                cp = gcode.pContour( self, pa, pb, pc, cr )   
                #---check if intersection point is possible---
                if cp["arcDir"] != "":
                    #---save intersection points of parallel lines---
                    if   cutpath == "Left":
                        xyCPath.append([cp["isecLX"],cp["isecLY"]])
                    elif cutpath == "Right":
                        xyCPath.append([cp["isecRX"],cp["isecRY"]])
            xyInterp = xyCPath

            
        #########################################################################
        #    02.07 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )
        
        
        #########################################################################
        #    02.08 g-code generation for module                                 #
        #########################################################################

        #---insert names and move to start position---
        for pName in pNames:
           code += pName 
           
        code += "G0 Z{:.3f}\n".format( thick+zSave )
        code += "G0 X{:.3f} Y{:.3f}\n".format( xyInterp[0][0],xyInterp[0][1] )

        #---loop until final depth was reached---
        currentdepth = 0.00
        while currentdepth < depth:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---update depth---
            code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code += "F{:.0f}\n".format( f )
            
            #---move to positions---
            for p in xyInterp:
                #---create gcode---
                code += "G1 X{:.3f} Y{:.3f}\n".format( p[0], p[1] )
            code += "\n"
            
            #---change direction---
            xyInterp.reverse()

        #---move z up---        
        code += "G0 Z" + str(thick+zSave) + "\n\n"


        #########################################################################
        #    02.09 end codes                                                    #
        #########################################################################

        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )

        return code 
   

   
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)


    #############################################################################
    #                                                                           #
    #    04.00 point is on line-segment                                         #
    #                                                                           #
    #############################################################################
    def pointOnLine( self, lPt1, lPt2, pt3):

        #---define points---
        x1,y1 = lPt1
        x2,y2 = lPt2
        x3,y3 = pt3

        #---calculate slope and check if it is on the line---
        slope = (y2-y1) / (x2-x1)
        pt3_on = (y3-y1) == slope * (x3-x1)

        #---check if point 3 is between point 1 and 2---
        pt3_between = (min(x1,x2) <= x3 <= max(x1,x2)) and (min(y1,y2) <= y3 <= max(y1,y2))          

        return pt3_on and pt3_between

#