'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Pocket Rectangle                                       #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Pocket - Rectangle"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "PocketRectangle.png"

#---explaining text for helpbutton in lonnox cut---
info = "PocketRectangleInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","O","N","N","N","N","N","N","N","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X Y Mode","X","Y",
             "Width (a)", "Height (b)", "Radius (r)", "Rotation", "Depth", "Cutdepth",
             "Overlap in %", "Feed")

#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror"),
             "X Y Mode":         ("X & Y Min","Center")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 
        
        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code         = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        mode          = jParams[jIndex][3]
        x             = jParams[jIndex][4] # + x0  see rotation calculation                      
        y             = jParams[jIndex][5] # + y0  see rotation calculation                     
        w             = jParams[jIndex][6]                      
        h             = jParams[jIndex][7]                      
        er            = math.fabs( jParams[jIndex][8] )         
        rotation      = jParams[jIndex][9]
        depth         = math.fabs( jParams[jIndex][10] ) 
        cutdepth      = math.fabs( jParams[jIndex][11] ) 
        overlap       = math.fabs( jParams[jIndex][12] ) 
        f             = jParams[jIndex][13]                     
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1

        #---set preview values---
        if preview: depth=1; cutdepth=1;        

        #---escape if essetial values are missing---
        if( not depth or not cutdepth or not w or not h or not cr or 
           (overlap < 0.0) or (overlap > 100.0)): 
            return code

  
        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################

        #---convert middle x y mode to edge x y mode---
        if mode == "Center":
            x = x - (w / 2)
            y = y - (h / 2)
    
        #---if edge radius is to small resize it to cutter radius---
        if er < cr : er = cr

        #---build gcode for inner path---
        cutP = [[ 1, x+er  , y     , f  ],
                [ 2, x     , y+er  , er ],
                [ 1, x     , y+h-er, f  ],
                [ 2, x+er  , y+h   , er ],
                [ 1, x+w-er, y+h   , f  ],
                [ 2, x+w   , y+h-er, er ],
                [ 1, x+w   , y+er  , f  ],
                [ 2, x+w-er, y     , er ]]

        #---recalculate mirror values---
        #if orient == "Mirror":
        #    for i in range(8):
        #        cutP[i][1] = mirrorstops-cutP[i][1]              
        #        if cutP[i][0] == 2: cutP[i][0] = 3              


        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )
        
        
        #########################################################################
        #    02.05 g-code generation for module                                 #
        #########################################################################

        #---move to rotated start position--- 
        code += "G0 Z{:.3f}\n".format( thick+zSave ) 
        rr = gcode.distancePP( self, (0,0), (cutP[7][1],cutP[7][2]+cr) )
        ra = rotation + gcode.pointAngle360( self, 0,0, cutP[7][1],cutP[7][2]+cr, metric=1 )
        if ra > 360: ra -= 360
        xStart,yStart = gcode.circlePoint( self, x0, y0, ra, rr ) 
        if orient == "Mirror": xStart = mirrorstops-xStart
        code += "G0 X{:.3f} Y{:.3f}\n".format(xStart,yStart)
        
        #---loop until final depth was reached---
        currentdepth = 0.00
        while 1:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth
            code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code += "G1 X{:.3f} Y{:.3f} F{:.0f}\n".format(xStart,yStart,f)

            dist = cr
            while 1:
                #---calculate path---
                if dist >= er:
                    cutP = [[ 1, x+dist  , y+dist  , f ], 
                            [ 0, x+dist  , y+dist  , 0 ],
                            [ 1, x+dist  , y+h-dist, f ],
                            [ 0, x+dist  , y+h-dist, 0 ],
                            [ 1, x+w-dist, y+h-dist, f ],
                            [ 0, x+w-dist, y+h-dist, 0 ],
                            [ 1, x+w-dist, y+dist  , f ],
                            [ 0, x+w-dist, y+dist  , 0 ]]
                else:
                    cutP = [[ 1, x+er    , y+dist  , f       ], 
                            [ 2, x+dist  , y+er    , er-dist ],
                            [ 1, x+dist  , y+h-er  , f       ],
                            [ 2, x+er    , y+h-dist, er-dist ],
                            [ 1, x+w-er  , y+h-dist, f       ],
                            [ 2, x+w-dist, y+h-er  , er-dist ],
                            [ 1, x+w-dist, y+er    , f       ],
                            [ 2, x+w-er  , y+dist  , er-dist ]]

                #---rotate rectangle points of cut path---
                cutR = []
                for i,(pg,px,py,pfr) in enumerate(cutP):
                    rr = gcode.distancePP( self, (0,0), (px,py) )
                    ra = rotation + gcode.pointAngle360( self, 0,0, px,py, metric=1 )
                    if ra > 360: ra -= 360
                    rx,ry = gcode.circlePoint( self, x0, y0, ra, rr ) 
                    cutR.append([pg,rx,ry,pfr])

                #---recalculate mirror values---
                if orient == "Mirror":
                    for i in range(8):
                        cutR[i][1] = mirrorstops-cutR[i][1]              
                        if cutR[i][0] == 2: cutR[i][0] = 3              

                #---build gcode steps---
                code += "G1 X{:.3f} Y{:.3f} F{:.0f}\n".format(cutR[7][1],cutR[7][2],f)
                for rg,rx,ry,rfr in cutR:
                    if rg==1: code += "G1 X{:.3f} Y{:.3f} F{:.0f}\n".format(rx,ry,rfr) 
                    if rg==2: code += "G2 X{:.3f} Y{:.3f} R{:.3f}\n".format(rx,ry,rfr) 
                    if rg==3: code += "G3 X{:.3f} Y{:.3f} R{:.3f}\n".format(rx,ry,rfr) 
                        
                #---if final inner path was reached end while loop---
                if dist >= (h/2-cr) or dist >= (w/2-cr): break             

                #---calculate next inner path---
                dist += cr * 2 - cr * 2 / 100 * overlap

            #---if final depth was reached then end while loop---
            if currentdepth >= depth: break;
            
        #---put rectangle cut and last segment cut together---
        code += "G0 Z" + str(thick+zSave) + "\n\n"
                  

        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )

        return code 
   


    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

