'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module HoleGrid                                               #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Drill"

#---name of the module in the tree---
name       = "HoleGrid"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "HoleGrid.png"

#---explaining text for helpbutton in lonnox cut---
info = "HoleGridInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","N","N","N","N","N","N","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X","Y","Xd",
             "Yd","Xn","Yn","Depth","Cutdepth","Plunge feed")
             
#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 
        
        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        x             = jParams[jIndex][3] + x0 
        y             = jParams[jIndex][4] + y0
        xdis          = jParams[jIndex][5] 
        ydis          = jParams[jIndex][6] 
        xloops        = int( jParams[jIndex][7] )
        yloops        = int( jParams[jIndex][8] )
        depth         = math.fabs( jParams[jIndex][9] )
        cutdepth      = math.fabs( jParams[jIndex][10] )
        depthfeed     = jParams[jIndex][11] 
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1

        #---set preview values---
        if preview: depth=1; cutdepth=1
        
        #---escape if essetial values are missing---
        if not depth or not cutdepth or not xdis or not ydis: return code
 
 
        #########################################################################
        #    02.02 mirror module on x-axis                                      #
        #########################################################################

        #---recalculate mirror values---
        if orient == "Mirror":
            x = mirrorstops - (x+(xloops-1)*xdis)


        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )

                     
        #########################################################################
        #    02.04 g-code generation for module                                 #
        #########################################################################

        #---loop for every x loop---
        dirtoggle = 0
        for ix in range(xloops):

            #---move y in zig-zag mode---   
            dirtoggle = not dirtoggle

            #---loop for every y loop---
            for iy in range(yloops):
                
                #---loop until final depth was reached---
                currentdepth = 0.00
                while currentdepth < depth:

                    #---set next depth level or final depth---
                    currentdepth += cutdepth
                    if currentdepth > depth: currentdepth = depth

                    #---start position---
                    if dirtoggle: iy2 = iy 
                    else: iy2 = (yloops-1)-iy
                    code += "G0 Z{0}\n".format( str(thick+zSave) )
                    code += "G0 X{0} Y{1}\n".format( str(x+(ix * xdis)), str(y+(iy2 * ydis)) )
                    
                    #---with depthfeed to depth---
                    code += "G1 Z{0} F{1}\n".format( str(thick-currentdepth), str(depthfeed) )

            #---end position---
            code += "G0 Z{0}\n\n".format( str(thick+zSave) )

                 
        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )

        return code 
   

 
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

         