'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Facing                                                 #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Facing"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "Facing.png"

#---explaining text for helpbutton in lonnox cut---
info = "FacingInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","N","N","N","N","N","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X","Y","Width (a)",
             "Height (b)","Depth","Cutdepth","Overlap in %","Feed")

#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 

        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################
        
        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        x             = jParams[jIndex][3] + x0                     
        y             = jParams[jIndex][4] + y0                     
        w             = math.fabs( jParams[jIndex][5] )         
        h             = math.fabs( jParams[jIndex][6] )         
        depth         = math.fabs( jParams[jIndex][7] )
        cutdepth      = math.fabs( jParams[jIndex][8] ) 
        overlap       = math.fabs( jParams[jIndex][9] )         
        f             = jParams[jIndex][10]                     
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
        
        #---set preview values---
        if preview: depth=1; cutdepth=1        
        
        #---if not tool is choosen show message---
        if not cr: uni.moduleStatus = L["Invalid tool for facing"]

        #---escape if essetial values are missing---
        if (not depth or not cutdepth or not w or 
            not h or not cr or (overlap >= 100)) : return code


        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################
         
        #---recalculate mirror values---
        if orient == "Mirror": x = mirrorstops - x - w

        #---set area to inner rectangle---
        x += cr
        y += cr
        w -= 2*cr
        h -= 2*cr
        
        #---determine y values--- 
        yvalues = [y+h,y]
        
        #---determine x values--- 
        xvalues = []
        paths = w / ((cr*2)-(cr*2*overlap/100)) 
        distance = w / paths
        for i in range( math.floor(paths+1) ):
            xvalues.append( x + (i * distance) ) 
        if xvalues[-1] != x+w: xvalues.append( x+w )
        
        
        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )

        
        #########################################################################
        #    02.04 g-code generation for module                                 #
        #########################################################################

        #---move to start position---
        code += "Z{}\n".format( thick+zSave )
        code += "G0 X{} Y{}\n".format( x, y )
        
        #---loop until final depth was reached---
        currentdepth = 0.00
        while currentdepth < depth:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---move to next depth level---
            code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code += "F{:.0f}\n".format( f )
        
            #---build gcode---  
            for xvalue in xvalues:  
                code += "G1 X{:.3f}\n".format( xvalue )
                code += "G1 Y{:.3f}\n".format( yvalues[0] )
                #---swap y values after every path--- 
                yvalues.reverse()
                
            #---reverse x values after every depth level---    
            xvalues.reverse()
            
        #---move z up---
        code += "G0 Z" + str(thick+zSave) + "\n\n"
                  

        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )

        return code 
   

   
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

        