'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Ellipse                                                #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Ellipse"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "Ellipse.png"

#---explaining text for helpbutton in lonnox cut---
info = "EllipseInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","N","N","N","N","N","N","N","N","N","O","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X","Y","Width (a)",
             "Height (b)","Rotation","Startangle","Endangle",
             "Depth","Cutdepth","Cutpath","Feed")

#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror"),
             "Cutpath":          ("Center","Inside","Outside")}
             #"Last segment":     ("Y Min","X Max","Y Max","X Min")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 

        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        x             = jParams[jIndex][3] + x0                     
        y             = jParams[jIndex][4] + y0                    
        xr            = math.fabs( jParams[jIndex][5] ) /2     
        yr            = math.fabs( jParams[jIndex][6] ) /2     
        subdiv        = 2000
        rotation      = jParams[jIndex][7]
        sAngle        = jParams[jIndex][8]                            
        eAngle        = jParams[jIndex][9]                              
        depth         = math.fabs( jParams[jIndex][10] )
        cutdepth      = math.fabs( jParams[jIndex][11] )
        cutpath       = jParams[jIndex][12]
        f             = jParams[jIndex][13]
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
        
        #---set preview values---
        if preview: depth=1; cutdepth=1; subdiv=100        

        #---escape if essetial values are missing---
        if not xr or not yr or not subdiv or not depth or not cutdepth: return code
        if math.fabs(eAngle-sAngle) == 0: return code
        
  
        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################
        
        #---set/toggle cutter compensation value---
        if cutpath == "Inside":
            #---for proper inner path minimum edge radius must be cutter radius---
            if cr > xr : xr = cr + 0.001
            if cr > yr : yr = cr + 0.001
            cr = -cr 
        if cutpath == "Center": cr = 0

        #---calculate ellipse coordinates---
        #Details: https://mathepedia.de/Parameterdarstellung.html
        angleRange = math.fabs(eAngle-sAngle)
        coords = []
        for i in range(subdiv):
            theta = angleRange / subdiv * i 
            theta += sAngle
            xx = x + ((xr+cr) * math.cos( math.radians(theta)))
            yy = y + ((yr+cr) * math.sin( math.radians(theta)))
            coords.append( [xx,yy] )

        #---rotate coords---
        rad = math.radians(rotation)
        coordsTemp = []
        for xx,yy in coords:
            xx -= x; yy -= y
            qx = math.cos(rad) * xx - math.sin(rad) * yy
            qy = math.sin(rad) * xx + math.cos(rad) * yy
            coordsTemp.append([x+qx,y+qy]) 
        coords = coordsTemp
       
        #---mirror the circle segments---
        if orient == "Mirror": 
            for i in range(len(coords)):
                coords[i][0] = mirrorstops - coords[i][0]
        
            
        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---startcode gcode module---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )

        
        #########################################################################
        #    02.04 g-code generation for module                                 #
        #########################################################################

        #---move to start position---
        code += "G0 Z{:.3f}\n".format( thick+zSave )
        code += "G0 X{:.3f} Y{:.3f}\n".format( coords[0][0], coords[0][1] )
        
        #---loop until final depth was reached---
        currentdepth = 0.00
        while currentdepth < depth:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---update depth---
            code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code += "F{:.0f}\n".format( f )

            #---build gcode---
            for coord in coords:
                code += "G1 X{:.3f} Y{:.3f}\n".format( coord[0], coord[1] )

            #---change direction---
            coords.reverse()
            
        #---put last segments and main cuts together---
        code += "G0 Z{:.3f}\n\n".format( thick + zSave )


        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )
        
        return code 



    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

                
