'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Contour Group                                          #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 change values by options                                         #
#        02.03 calculate cutpath and open/close options                         #
#        02.04 calculate tangential roundings                                   #
#        02.05 create list for forth and back direction                         #
#        02.06 add colsed contour position                                      #
#        02.07 startcodes                                                       #
#        02.08 g-code generation for module                                     #
#        02.09 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni
import csv


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Contour-Group"
groupEnd   = "Contour-End"
 
#---explain picture of the module--- 
picture    = "ContourG.png"

#---explaining text for helpbutton in lonnox cut---
info = "ContourGInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","O","O","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","Contour Type",
             "Cutpath","Depth","Cutdepth","Feed")

#---option presets for listed parameters---
options   = {"Orientation":        ("Normal","Mirror"), 
             "Contour Type":       ("Open","Close"), 
             "Cutpath":            ("Center","Left","Right") }



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 

        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        conType       = jParams[jIndex][3] 
        cutpath       = jParams[jIndex][4] 
        depth         = math.fabs( jParams[jIndex][5] )
        cutdepth      = math.fabs( jParams[jIndex][6] )
        f             = jParams[jIndex][7]   
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
        
        #---search and collect contour points---
        xyr = []  #PointX,PointY,PointRadius
        pNames = [] #PointNames
        i = jIndex+1
        while i < len(jNames): 
            p = jParams[i]
            #---search contour-ponts in jobnames---
            if jNames[i][:-4] == "Contour-Point": 
                #---add point---
                xyr.append( [x0+p[0]+p[2], y0+p[1]+p[3], p[4]] )
                pNames.append( "." + L["Point"][:1] + jNames[i][-3:] )
            #---search contour-files in jobnames---
            elif jNames[i][:-4] == "Contour-List":
                        #---build contour list---
                        xL,yL,rL = p[2].split(";"),p[3].split(";"),p[4].split(";")
                        #---iterate through rows---
                        j = 0
                        while j<len(xL) and j<len(yL):
                            #---set r to 0.0 if not available--- 
                            try: r = float(rL[j].replace(",","."))
                            except (IndexError,ValueError): r = 0.0   
                            #---skip if x or y not available---
                            try:
                                x = float(xL[j].replace(",","."))
                                y = float(yL[j].replace(",","."))
                                #---add Point---
                                xyr.append( [x0+p[0]+x, y0+p[1]+y, r] )
                                pNames.append( L["Point"][:1] + "{:03d}".format(j) )
                            except: pass
                            j += 1    
            #---cancel on first "non-contour-job"---
            else: break 
            i += 1
            
        #---set preview values---
        if preview: depth=2; cutdepth=1
        
        #---escape if essetial values are missing---
        if not depth or not cutdepth or len(xyr) < 2: return code


        #########################################################################
        #    02.02 change values by options                                     #
        #########################################################################
        
        #---mirror points---
        if orient == "Mirror":
            for val in xyr: val[0] = mirrorstops - val[0]
            if   cutpath == "Left" : cutpath = "Right"
            elif cutpath == "Right": cutpath = "Left"
        
        #---build point names after mirrorring--- 
        for i,(x,y,r) in enumerate(xyr):
            pNames[i]= "({} X{} Y{})\n".format(pNames[i],x,y)
            
          
        #########################################################################
        #    02.03 calculate cutpath and open/close options                     #
        #########################################################################
        
        #---set default coords---          
        xyxyr = [[val[0],val[1],0,0,val[2]] for index, val in enumerate(xyr)]

        #---insert start endpoints by path "Contour type"---
        if conType == "Close": 
            xyr.insert( 0, list(xyr[-1]) )        
            xyr.append( list(xyr[1]) )
        else:
            xyr.insert( 0, [xyr[0][0]+(xyr[0][0]-xyr[1][0]), xyr[0][1]+(xyr[0][1]-xyr[1][1]), 0] )        
            xyr.append( [xyr[-1][0]-(xyr[-2][0]-xyr[-1][0]), xyr[-1][1]-(xyr[-2][1]-xyr[-1][1]), 0] )
            xyr[0][2]=0; xyr[-1][2]=0        
            xyxyr[0][4]=0; xyxyr[-1][4]=0        
        
        #---recalculate path if left or right was choosen---
        if (cutpath != "Center"):
            for i in range(len(xyr)-2):
                #---get previous, current and next point---
                PA=xyr[i][:2]; PB=xyr[i+1][:2]; PC=xyr[i+2][:2]; r=xyr[i+1][2] 
                #---calculate parallel contour points---  
                cP = gcode.pContour( self, PA, PB, PC, cr )   
                #---compensate radius if offset is choosen---    
                if r > 0:
                    if cutpath[:1].lower() == cP["arcDir"]: r -= cr
                    else: r += cr
                #---save contour points---
                if (cP["arcDir"] == "") or (r < 0): r = 0
                #---save contour points by cutpath side---
                if cutpath == "Right":
                    xyxyr[i][0] = cP["isecRX"]
                    xyxyr[i][1] = cP["isecRY"]
                else:
                    xyxyr[i][0] = cP["isecLX"]
                    xyxyr[i][1] = cP["isecLY"]
                xyxyr[i][4] = r
                

        #########################################################################
        #    02.04 calculate tangential roundings                               #
        #########################################################################
        
        #---set default coords---          
        xyxyrg = [[val[0],val[1],0,0,val[4],""] for val in xyxyr]
                
        #---insert last element before first element---
        xyxyr.insert( 0, list(xyxyr[-1]) )        
        #---insert first element after last element---
        xyxyr.append( list(xyxyr[1]) )

        #---recalculate path if radius was choosen---
        for i in range(len(xyxyr)-2):
            #---get previous, current and next point---
            PA=xyxyr[i][:2]; PB=xyxyr[i+1][:2]; PC=xyxyr[i+2][:2]; r=xyxyr[i+1][4]   
            if (r > 0) and (PB != PA) and (PB != PC) and (PA != PC): 
                #---calculate parallel contour points---  
                cP = gcode.pContour( self, PA, PB, PC, r )   
                #---save contour points---
                xyxyrg[i][0] = cP["pASx"]
                xyxyrg[i][1] = cP["pASy"]
                xyxyrg[i][2] = cP["pCEx"]
                xyxyrg[i][3] = cP["pCEy"]
                xyxyrg[i][5] = cP["arcDir"]                


        #########################################################################
        #    02.05 create list for forth and back direction                     #
        #########################################################################
        
        #---forth direction--- 
        xyxyrgF = list(xyxyrg)
        
        #---back direction--- 
        xyxyrg.reverse()
        xyxyrgB = []         
        for p in xyxyrg:
            if p[4] > 0: 
                if   p[5] == "l": g = "r"
                elif p[5] == "r": g = "l"
                else: g = ""
                xyxyrgB.append( (p[2],p[3],p[0],p[1],p[4],g) ) 
            else: xyxyrgB.append( p )

            
        #########################################################################
        #    02.06 add colsed contour position                                  #
        #########################################################################

        #---add startposition as endposition for closed contour type---
        if conType == "Close":
            xyxyrgF.append( (xyxyrgF[0][0],xyxyrgF[0][1],0,0,0,"") )        

            
        #########################################################################
        #    02.07 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )
        
        
        #########################################################################
        #    02.08 g-code generation for module                                 #
        #########################################################################

        #---insert names and move to start position---
        for i,pName in enumerate(pNames):
           code += pName 
           
        code += "G0 Z{:.3f}\n".format( thick+zSave )
        code += "G0 X{:.3f} Y{:.3f}\n".format( xyxyrgF[0][0],xyxyrgF[0][1] )

        #---loop until final depth was reached---
        dirtoggle = 0 
        currentdepth = 0.00
        while currentdepth < depth:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---update depth---
            code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code += "F{:.0f}\n".format( f )
            
            #---change direction---
            if not dirtoggle: xyxyrgFB = xyxyrgF
            else: xyxyrgFB = xyxyrgB
            dirtoggle ^= 1

            #---move to positions---
            for p in xyxyrgFB:
                #---create gcode---
                code += "G1 X{:.3f} Y{:.3f}\n".format( p[0], p[1] )
                if p[5] == "l":
                    code += "G3 X{:.3f} Y{:.3f} R{:.3f}\n".format( p[2], p[3], p[4] )
                elif p[5] == "r":
                    code += "G2 X{:.3f} Y{:.3f} R{:.3f}\n".format( p[2], p[3], p[4] )
            code += "\n"
            
        #---move z up---        
        code += "G0 Z" + str(thick+zSave) + "\n\n"


        #########################################################################
        #    02.09 end codes                                                    #
        #########################################################################

        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )

        return code 
   

   
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

