'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Bow                                                    #
#                                                                               #
#################################################################################
#	C O N T E N T                                								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Bow"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "Bow.png"

#---explaining text for helpbutton in lonnox cut---
info = "BowInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","N","N","N","N","N","N","N","O","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X","Y","Radius (r)","Startangle",
             "Endangle","Depth","Cutdepth","Cutpath","Feed")

#---option presets for listed parameters---
options   = {"Orientation":     ("Normal","Mirror"),
             "Cutpath":         ("Center","Inside","Outside")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 

        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels

        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        x             = jParams[jIndex][3] + x0                             
        y             = jParams[jIndex][4] + y0                             
        r             = math.fabs( jParams[jIndex][5] )                 
        sa            = jParams[jIndex][6]                            
        ea            = jParams[jIndex][7]                              
        depth         = math.fabs( jParams[jIndex][8] )
        cutdepth      = math.fabs( jParams[jIndex][9] )
        cutpath       = jParams[jIndex][10]
        f             = jParams[jIndex][11]                              
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
        
        #---set preview values---
        if preview: depth=1; cutdepth=1
        
        #---escape if essetial values are missing---
        if not r or not sa-ea or not depth or not cutdepth: return code

		
        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################
         
        #---convert angle values to 0-360° values---
        while sa < 0 : sa += 360 
        while sa >= 360: sa -= 360         
        while ea < 0 : ea += 360 
        while ea >= 360: ea -= 360         

        #---set/toggle cutter compensation value---
        if cutpath == "Inside":
            #---for proper inner path minimum edge radius must be cutter radius---
            if cr > r : r = cr + 0.001
            cr = -cr 
        if cutpath == "Center": cr = 0

        #---calculate circle coords on start and end angle---
        r = r+cr 
        xs = math.cos( math.radians(sa) ) * r  
        ys = math.sin( math.radians(sa) ) * r 
        xe = math.cos( math.radians(ea) ) * r  
        ye = math.sin( math.radians(ea) ) * r 

        #---check if segment size is more then 180° and G2/3 need neg sign---
        if ea > sa:
            if (ea-sa) > 180: r = -r
        else:
            if (360-sa)+ea > 180: r = -r 
        
        #---mirror the circle segment---
        if orient == "Mirror":
            x = mirrorstops - x
            #---change sign of the x values--- 
            xs=-xs; xe=-xe;
            #---swap start and end point---
            temp=xs; xs=xe; xe=temp
            temp=ys; ys=ye; ye=temp


        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )


        #########################################################################
        #    02.04 g-code generation for module                                 #
        #########################################################################

        #---move to start position---
        code += "G0 Z{:.3f}\n".format( thick+zSave )
        code += "G0 X{:.3f} Y{:.3f}\n".format( x+xs, y+ys )
        
        #---loop until final depth was reached---
        dirtoggle = 0 
        currentdepth = 0.00
        while 1:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---update depth for rectangle cut and last segment cut---
            code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
        

            #---build steps for counterclock wise cut direction---  
            if not dirtoggle:
                code += "G3 X{:.3f} Y{:.3f} R{:.3f} F{:.0f}\n".format( x+xe, y+ye, r, f )

            #---build steps for clock wise cut direction---  
            else:
                code += "G2 X{:.3f} Y{:.3f} R{:.3f} F{:.0f}\n".format( x+xs, y+ys, r, f )
            dirtoggle ^= 1

            #---if final depth was reached then end while loop---
            if currentdepth == depth: break;
            
        #---put rectangle cut and last segment cut together---
        code += "G0 Z" + str(thick+zSave) + "\n\n"
                  

        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )

        return code 
   

   
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

           
   