'''
#################################################################################
#                                                                               #
#    G-Code Library                                                             #
#                                                                               #
#################################################################################
# Copyright 2020 Kai Masemann                                                   #
#################################################################################
# This file is part of Lonnox CUT                                               #
#                                                                               #
#   Lonnox CUT is free software: you can redistribute it and/or modify          #
#   it under the terms of the GNU General Public License as published by        #
#   the Free Software Foundation, either version 3 of the License, or           #
#   (at your option) any later version.                                         #
#                                                                               #
#   Lonnox CUT is distributed in the hope that it will be useful,               #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of              #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
#   GNU General Public License for more details.                                #
#                                                                               #
#   You should have received a copy of the GNU General Public License           #
#   along with Lonnox CUT.  If not, see <http://www.gnu.org/licenses/>.         #
#                                                                               #
#################################################################################
#	C O N T E N T	     					                                    #
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 Load Librarys                                                        #
#                                                                               #
#    01.00 Global Variables                                                     #
#                                                                               #
#    02.00 Start Code                                                           #
#                                                                               #
#    03.00 End Code                                                             #
#                                                                               #
#    04.00 CSV Tool                                                             #
#                                                                               #
#    05.00 List Rotate                                                          #
#                                                                               #
#    06.00 Find Tool Number From Last Toolchange                                #
#                                                                               #
#    07.00 Convert G2/G3 Values To QPainterPath.arcTo Values                    #
#        07.01 i and j values are known                                         #
#        07.02 r value is known                                                 #
#                                                                               #
#    08.00 calculate angle of a points on a Circle (0-360°)                     #
#                                                                               #
#    9.00 extract x and y from g-code string                                    #
#                                                                               #
#   10.00 extract x,y,w and h from g-code string for rectangles                 #
#                                                                               #
#   11.00 calculate intersection point of 2 lines                               #
#                                                                               #
#   12.00 calc points for parallel contours or tangential rounding              #
#                                                                               #
#   13.00 calculate coords of a point on a circle                               #
#                                                                               #
#   14.00 calculate the distance between two points                             #
#                                                                               #
#   15.00 calculate the distance from point to line                             #
#                                                                               #
#   16.00 tool de-/activation optimizer                                         #
#                                                                               #
#   17.00 calculate angle for font bows                                         #
#                                                                               #
#   18.00 extract values from dxf layer                                         #
#                                                                               #
#   19.00 convert dxf radius to contour radius                                  #
#                                                                               #
#   20.00 calculate the intersection between a line and a circle                #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <k.masemann@lavalu.de>'

#################################################################################
#                                                                               #
#    00.00 Load Libraries                                                       #
#                                                                               #
#################################################################################

#---Libraries For The Layout---
import math
import os
import sys
import csv
import uni


#################################################################################
#                                                                               #
#    01.00 Gloabl Variables                                                     #
#                                                                               #
#################################################################################

#---File And Dirnames---
if getattr(sys, 'frozen', False):
    # frozen
    path = os.path.dirname(sys.executable)
else:
    # unfrozen
    path = os.path.dirname(os.path.realpath(__file__))
#path = os.path.dirname(os.path.abspath(__file__))


#############################################################################
#                                                                           #
#    02.00 Start Code                                                       #
#                                                                           #
#############################################################################
def start( self, thick ):
    L = uni.language
    
    #---Startcode For Gcode Files---
    code  = ("G90" + (" "*22) + L["(Use absolute coordinates)\n"] +
             "G21" + (" "*22) + L["(Dimensionvalues in mm)\n"] +
             "G17" + (" "*22) + L["(XY plane)\n"] +
             "G94" + (" "*22) + L["(Speedvalues in mm/Min)\n"] +
             "G40" + (" "*22) + L["(Cancel radius compensation)\n"] +
             "G64 P0.1" + (" "*17) + L["(Curve tolerance in mm)\n"] +
             "\n" )
    
    return code 


#############################################################################
#                                                                           #
#    03.00 End Code                                                         #
#                                                                           #
#############################################################################
def end( self ):

    #---Endcode For Gcode Files---
    code  = ("\n")
    
    return code 



#############################################################################
#                                                                           #
#    04.00 CSVTool                                                          #
#                                                                           #
#############################################################################
def csvtool( self, toolno, toolManual=["","","",0,0,0,0] ):
    
    #---Load Mcodes From tools.csv---
    if toolno != "100":
        codes = {}
        for no,text,on,off,m6,g43,dia,cdir,typ,ico in uni.toollist:
            #---Load Tool Codes---
            if no == toolno:
                codes = {"on":on, "off":off, "m6":m6, "g43":g43}
                if dia[0].upper() == "R": dia = dia[1:]
                try: codes["cr"]  = float(dia)
                except ValueError: pass
                    
        #---set default values in unchangeble code dictonary--- 
        keys = ["on","off","m6","g43","cr"]
        for k in keys:
            if not k in codes:
                if k == "cr": codes[k] = 0.00
                else: codes[k] = ""            
    
    #---Load Manual Tool Data---
    else:
        codes = {"on" :toolManual[1], 
                   "off":toolManual[2], 
                   "m6" :"M6 T0", 
                   "g43":"G43.1 X{} Y{} Z{}".format(toolManual[3],toolManual[4],toolManual[5]),
                   "cr" : toolManual[6]/2}
    
    return codes
     
         
    

#############################################################################
#                                                                           #
#    05.00 Lsit Rotate                                                      #
#                                                                           #
#############################################################################
def rotate( self, array, rotation ):

    #---Rotate List By Rotation Value---
    array = array[rotation:] + array[:rotation]
    
    return array


#############################################################################
#                                                                           #
#    06.00 Find Tool Number From Last Toolchange                            #
#                                                                           #
#############################################################################
def findToolchange( self, joblist, jIndex ):

    #---search for toolchange in joblist (except its disabled)---
    tool = ""
    i = 1
    while jIndex >= i:
        job = joblist[jIndex-i]
        if (job[1][:-4] == "Toolchange") or (job[1][:-4] == "Tool - manual"):
            tool = job[4:]
            break;
        i += 1    
    
    return tool

    

#############################################################################
#                                                                           #
#    07.00 Convert G2/G3 Values To QPainterPath.arcTo Values                #
#                                                                           #
#############################################################################
def g23Convert( self, g, xStart, yStart, xEnd, yEnd, i=0, j=0, r=0 ):
    
    #########################################################################
    #    07.01 i and j values are known                                     #
    #########################################################################
    if r == 0:

        #---calculate the center point---
        xc = xStart + i
        yc = yStart + j

        #---calculate the radius---
        r = math.sqrt(math.fabs(i)**2 + math.fabs(j)**2) 
        d = r * 2

        #---calculate rectangle origin---
        xRec = xc - r
        yRec = yc - r
 
        #---calculate start/ end angle--- 
        aStart = pointAngle360(self, xc, yc, xStart, yStart) 
        aEnd   = pointAngle360(self, xc, yc, xEnd, yEnd) 
        
        #---swap start and end angle for G2---
        if g == "G2": aSwap = aStart; aStart = aEnd; aEnd = aSwap 
        
        #---calculate angle sapn---
        if aStart < aEnd: span = aEnd - aStart
        else: span = 360 - aStart + aEnd

        

    #########################################################################
    #    07.02 r value is known                                             #
    #########################################################################
    else:

        #---if calculation is impossible return---
        if xStart == xEnd and yStart == yEnd: return(0,0,0,0,0,0)

        #---calculate the angle alpha from the linie between start and endpoint---
        rAbs = math.fabs(r)
        b1 = xEnd - xStart
        a1 = yEnd - yStart
        c1 = math.sqrt(a1**2 + b1**2)
        alpha1 = pointAngle360( self, xStart, yStart, xEnd, yEnd )
        
        #---calculate the angle from start-end-line to circle intersection point---
        b2 = c1 / 2
        c2 = r
        a2 = math.sqrt(math.fabs(c2**2 - b2**2))
        q2 = a2**2 / c2
        p2 = c2 - q2
        h2 = math.sqrt(p2*q2)
        alpha2 = math.degrees( math.atan(h2/p2) )
        
        #---calculate start and end angles for start and end points--- 		
        aEnd1 = alpha1 - alpha2
        if aEnd1 < 0: aEnd1 += 360
        aEnd2 = alpha1 + alpha2
        if aEnd2 >= 360: aEnd2 -= 360
        aStart1 = aEnd2 - 180 
        if aStart1 < 0: aStart1 += 360
        aStart2 = aEnd1 - 180
        if aStart2 < 0: aStart2 += 360
		
        #---calculate coords from two intersection points---
        xc1 = (math.cos( math.radians( aEnd2 ) ) * rAbs) + xStart
        yc1 = (math.sin( math.radians( aEnd2 ) ) * rAbs) * (-1) + yStart 
        xc2 = (math.cos( math.radians( aEnd1 ) ) * rAbs) + xStart
        yc2 = (math.sin( math.radians( aEnd1 ) ) * rAbs) * (-1) + yStart

        #---swap start and end angle for G2---
        if g == "G2": 
            aSwap1 = aStart1; aStart1 = aEnd1; aEnd1 = aSwap1 
            aSwap2 = aStart2; aStart2 = aEnd2; aEnd2 = aSwap2 
        
        #---calculate angle sapn---
        if aStart1 < aEnd1: span1 = aEnd1 - aStart1
        else: span1 = 360 - aStart1 + aEnd1
        if aStart2 < aEnd2: span2 = aEnd2 - aStart2
        else: span2 = 360 - aStart2 + aEnd2

        #---calculate d---
        d = 2 * rAbs

        #---filter by r sign and span degree---
        if r >= 0:
            if span1 < span2: xRec = xc1-rAbs; yRec = yc1-rAbs; aStart = aStart1; span = span1
            else: xRec = xc2-rAbs; yRec = yc2-rAbs; aStart = aStart2; span = span2
        else:
            if span1 > span2: xRec = xc1-rAbs; yRec = yc1-rAbs; aStart = aStart1; span = span1
            else: xRec = xc2-rAbs; yRec = yc2-rAbs; aStart = aStart2; span = span2

    return (xRec, yRec, d, d, aStart, span) 


    
#############################################################################
#                                                                           #
#    08.00 calculate angle of a points on a Circle (0-360°)                 #
#                                                                           #
#############################################################################
# x/yCenter: start point of the line
# x/yPoint: end point of the line
# metric: if set metric coordinatesystem is used else pixel system is used
# result: angle of the line   
def pointAngle360( self, xCenter, yCenter, xPoint, yPoint, metric=0 ):

    #---calculate the radius---
    i = xPoint-xCenter
    j = yPoint-yCenter
    r = math.sqrt(math.fabs(i)**2 + math.fabs(j)**2)

    #---calculate cos/ sin angles---
    if r == 0: return 0 
    aCos = math.degrees( math.acos(i/r) )
    aSin = math.degrees( math.asin(j/r) )

    #---quadrantfiltering depends on metric or pixel(x swap)---
    if metric:
        if aSin > 0: angle = aCos
        else: angle = 360 - aCos
    else:
        if aSin < 0: angle = aCos
        else: angle = 360 - aCos
    if angle == 360: angle = 0      
      
     
    return angle


	
#############################################################################
#                                                                           #
#    9.00 extract x and y from g-code string                                #
#                                                                           #
#############################################################################
def gExtract( self, line, coords, xmax ):

    #---reset gcodes that are not modal---
    coords[2] = coords[3] = coords[4] = coords[5] = coords[6] = 0
    
    #---strings to search---
    s = ["X","Y","I","J","R","P","Q"]
    
    #---search x in string---
    for i in range( len(s) ):
        first = line.find( s[i] )  
        if first >= 0:
            first += 1
            last = line.find(" ",first)
            if last < 0: last = len( line )
            #---extraxt value from string---
            coords[i] = float(line[first:last])
            #---mirror x values---
            if i == 0: coords[i] = xmax - coords[i]
            #---mirror i values---
            if i == 2: coords[i] = -coords[i]    
            #---mirror p values---
            if i == 5: coords[i] = -coords[i]    
    return  



#############################################################################
#                                                                           #
#   10.00 extract x,y,w and h from g-code string for rectangles             #
#                                                                           #
#############################################################################
def rExtract( self, line, xmax ):
    
    #---init coords---
    coords = [0,0,0,0]

    #---strings to search---
    s = [" X"," Y"," a"," b"]
    
    #---search x in string---
    for i in range( len(s) ):
        first = line.find( s[i] )  
        if first >= 0:
            first += 2
            last = line.find(" ",first)
            if last < 0: last = len( line )
            #---extraxt value from string---
            coords[i] = float(line[first:last])
    #---mirror x values---
    coords[0] = xmax - coords[0] - coords[2]

    return coords  


#############################################################################
#                                                                           #
#   11.00 calculate intersection point of 2 lines                           #
#                                                                           #
#############################################################################
def intersec( self, lineA, lineB ):
    '''
    #line data (x1,y1,x2,y2)
    '''
    #---check if intersection is possible---
    if (lineA[0]-lineA[2] == 0) and (lineB[0]-lineB[2] == 0): return (0,0,False)
    if (lineA[1]-lineA[3] == 0) and (lineB[1]-lineB[3] == 0): return (0,0,False)

    #---equalation lineA: y = m * x + n ---
    try: mA = (lineA[1]-lineA[3]) / (lineA[0]-lineA[2])
    except ZeroDivisionError: mA = 9e20
    nA = lineA[1] - mA * lineA[0]    

    #---equalation lineB: y = m * x + n ---
    try: mB = (lineB[1]-lineB[3]) / (lineB[0]-lineB[2])
    except ZeroDivisionError: mB = 9e20
    nB = lineB[1] - mB * lineB[0]    

    #---intersection equalation: lineA = lineB ---
    n = nB + (-nA)
    m = mA + (-mB)
    try: x = n / m
    except ZeroDivisionError: x = 9e20
    y = mA * x + nA

    #---precise values for infinetley calculations---
    if   lineA[0] == lineA[2]: x = lineA[0]
    elif lineB[0] == lineB[2]: x = lineB[0]
    if   lineA[1] == lineA[3]: y = lineA[1]
    elif lineB[1] == lineB[3]: y = lineB[1]

    return(x,y,True) 
 

#############################################################################
#                                                                           #
#   12.00 calc points for parallel contours or tangential rounding          #
#                                                                           #
#############################################################################
def pContour( self, pointA, pointB, pointC, offset ):
    '''
    #siehe auch: http://www.arndt-bruenner.de/mathe/scripts/dreiecksberechnungrw.htm
    #this function calculate the points that are needed for parallel 
    #countouring and tangential rounding of 2 conncted lines.
    #if r is 0 Start and Endpoint have the same value (intersection point)        
    #side have to be "l" for left and "r" for right.
    #resulting points:
    #pAS: Startpoint of the arc bow
    #pCE: Endpoint of the arc bow
    #isecL: left intersection point or centerpoint for left arc bows
    #isecR: right intersection poinnt or centerpoint for right arc bows 
    #arcDir: Indecates if arc bow is left right or not possible ("l"/"r"/"").    
    '''

    #---calculate intersection points isecR/L---
    angA = pointAngle360( self, pointB[0], pointB[1], pointA[0], pointA[1], 1)
    angC = pointAngle360( self, pointB[0], pointB[1], pointC[0], pointC[1], 1)
    if angA > angC: angAC = angA-angC 
    else: angAC = 360 - angC + angA
    if angAC >= 180: beta = (360 - angAC) / 2
    else: beta = angAC / 2
    if beta == 90 or not beta: a = 0; c = offset  
    else:
        a = math.tan( math.radians( 90-beta ) ) * offset            
        c = math.sqrt(a**2 + offset**2)
    angL = angA - (angAC/2)
    angR = angL - 180
    if angL < 0: angL += 360
    if angR < 0: angR += 360
    isecL = circlePoint( self, pointB[0], pointB[1], angL, c )                 
    isecR = circlePoint( self, pointB[0], pointB[1], angR, c )                 
    #---determine arcDir---
    if angAC > 180: arcDir = "r"
    elif angAC < 180: arcDir = "l"
    else: arcDir = ""
    #---calculate start (pAS) and endpoint (pCE) of the arc bow---    
    pAS = circlePoint( self, pointB[0], pointB[1], angA, a )                 
    pCE = circlePoint( self, pointB[0], pointB[1], angC, a )                 

    pC = {"pASx":pAS[0], "pASy":pAS[1], "pCEx":pCE[0], "pCEy":pCE[1], "arcDir":arcDir}
    pC["isecLX"] = isecL[0]
    pC["isecLY"] = isecL[1]
    pC["isecRX"] = isecR[0]
    pC["isecRY"] = isecR[1]
    
    return pC  
 

    
#############################################################################
#                                                                           #
#   13.00 calculate coords of a point on a circle                           #
#                                                                           #
#############################################################################
# x,y: coordinates of the circles center point
# angle: direction of the unknown point
# r: distance from the center to the searched point
# return: the absolute coordinates of the searched point
def circlePoint( self, x, y, angle, r ):

    #---calculate coords---     
    xa = math.cos( math.radians(angle) ) * r  
    ya = math.sin( math.radians(angle) ) * r 
    
    return(x+xa,y+ya)

    
    
#############################################################################
#                                                                           #
#   14.00 calculate the distance between two points                         #
#                                                                           #
#############################################################################
# pointA/B format: (x,y)
def distancePP( self, pointA, pointB ):
    
    #---distance with pythygoras---
    a = math.fabs(pointA[0] - pointB[0])
    b = math.fabs(pointA[1] - pointB[1])
    
    return math.sqrt( a**2 + b**2 )
    

    
#############################################################################
#                                                                           #
#   15.00 calculate the distance form a point to a line                     #
#                                                                           #
#############################################################################
# line: (x1,y1,x2,y2): line of distance calculation
# point: (x,y): point of distance calculation
# result: (distance, intersOnLine): shortest distance from line to point,
#                                   indicator for "intersection is between 
#                                   lineA/B points"
def distanceLP( self, line, point ):
    '''
    #line data (x1,y1,x2,y2) 
    '''
    
    #---get line angle---
    lineAng = pointAngle360( self, line[0], line[1], line[2], line[3], metric=1 )
    
    #---calculate right angled line---
    pointAng = lineAng + 90
    if pointAng >= 360: pointAng -= 360
    pointB = circlePoint( self, point[0], point[1], pointAng, 1 )
    
    #---calculate intersection---
    pointI = intersec( self, (line), (point[0],point[1],pointB[0],pointB[1]) )[:2]
    
    #---calculate line middle points---
    lineMX = line[0] + ( (line[2] - line[0]) / 2 )
    lineMY = line[1] + ( (line[3] - line[1]) / 2 )
    
    #---proof if intersection is on line---
    lineDis = distancePP( self, (line[0],line[1]), (line[2],line[3]) ) 
    if distancePP( self, (lineMX,lineMY), pointI ) <= (lineDis / 2): intersOnLine = True
    else: intersOnLine = False         
    
    #---calculate line to point distance---
    pointDis = distancePP( self, point, pointI ) 
    
    return (pointDis, intersOnLine)
    
 

#############################################################################
#                                                                           #
#   16.00 tool de-/activation optimizer                                     #
#                                                                           #
#############################################################################
def toolOptimizer( self, jIndex, jTools, csvT, onOff="on", search=True ):
    L = uni.language 
  
    code = ""
    i = 1
    if onOff == "on":
        #---search for previous (relevant) endtool---
        prevTool = ""; prevValues = [] #[tStaNr,tEndNr,tStaValues,tEndValues]
        while (i <= jIndex) and search:
            if jTools[jIndex-i][1] != "": 
                prevTool = jTools[jIndex-i][1]
                prevValues = jTools[jIndex-i][3]
                break 
            i += 1            
        
        #---only activate tool if necessary--- 
        if (prevTool != jTools[jIndex][0]) or (prevValues != jTools[jIndex][2]):
            code = (L["(Tool activate)\n"] + 
                    "G53 G0 Z{:.3f}\n".format( uni.settings[3] ) +
                    csvT["m6"] + "\n" +  
                    csvT["g43"] + "\n" +  
                    csvT["on"] + "\n\n")

    else:
        #---search for next (relevant) starttool---
        nextTool = ""; nextValues = [] #[tStaNr,tEndNr,tStaValues,tEndValues]
        while (jIndex+i) < len(jTools) and search:
            if jTools[jIndex+i][0] != "": 
                nextTool = jTools[jIndex+i][0]
                nextValues = jTools[jIndex+i][2]
                break 
            i += 1            
        
        #---only deactivate tool if necessary--- 
        if (nextTool != jTools[jIndex][1]) or (nextValues != jTools[jIndex][3]):
            #---Sleep After M5 To Prevent Missing Toolchanges---
            s = ""
            if "M5" in csvT["off"].upper():
                s += "\nG4 P" + str(uni.m5Wait)
			
            #---Tool Deactivation---
            code = (L["(Tool deactivate)\n"] +
                    "G53 G0 Z{:.3f}\n".format( uni.settings[3] ) +
                    csvT["off"] + s + "\n\n") 
    return code


#############################################################################
#                                                                           #
#   17.00 calculate angle for font bows                                     #
#                                                                           #
#############################################################################
#for details see http://www.arndt-bruenner.de/mathe/scripts/dreiecksberechnungrw.htm
def fontBow( self, bowRadius, glyphWidth ):
    
    #---check values---
    if not bowRadius or not glyphWidth: return 0
    
    #---calculate angle---
    q = (glyphWidth/2)**2 / bowRadius
    p = bowRadius - q
    h = math.sqrt(p*q)
    return math.degrees( math.atan( h / p ) )
    

#############################################################################
#                                                                           #
#    18.00 extract values from dxf layer                                    #
#                                                                           #
#############################################################################
def dxfExtract( self, layer ):

    #---strings to search---
    s = ["_T","_F","_P","_C","_W"]
    names = ["tool","feed","cutpath","cutdepth","bladewidth"]
    params = {}
    
    #---search x in string---
    for i in range( len(s) ):
        first = layer.find( s[i] )  
        if first >= 0:
            first += 2
            last = layer.find("_",first)
            if last < 0: last = len( layer )
            params[ names[i] ] = layer[first:last]    
        else: params[ names[i] ] = "" 
    
    return  params


#############################################################################
#                                                                           #
#    19.00 convert dxf radius to contour radius                             #
#                                                                           #
#############################################################################
#for details see http://www.arndt-bruenner.de/mathe/scripts/kreissehnen.htm
def dxfContourRadius( self, pA, pB, bow ):
    #pA : Startpoint of the bow (x,y) 
    #pB : Endpoint of the bow (x,y) 
    #bow  : float with dxf group code 42	
    
    #---calculate radius--- 
    if bow < 0: bowR = 1
    else: bowR = 0
    dis = distancePP( self, pA, pB )
    alpha = math.degrees( math.atan( math.fabs(bow) ) ) * 4
    r = dis/(2*math.sin( math.radians( alpha ) /2) )    
    
    #---calculate middle point of circle and filter results by angle and direction---
    if bowR:
        mx = 0.5 * (pA[0]+pB[0] + (1/math.tan( math.radians(alpha/2)) ) * (-pA[1]+pB[1]))
        my = 0.5 * (pA[1]+pB[1] + (1/math.tan( math.radians(alpha/2)) ) * (pA[0]-pB[0]))
    else:
        mx = 0.5 * (pA[0]+pB[0] - (1/math.tan( math.radians(alpha/2)) ) * (-pA[1]+pB[1]))
        my = 0.5 * (pA[1]+pB[1] - (1/math.tan( math.radians(alpha/2)) ) * (pA[0]-pB[0]))
    
    #---calculate new points---
    if alpha > 240: subs = 6
    elif alpha > 120: subs = 4
    else: subs = 2
    angS = alpha/subs
    
    angA = pointAngle360( self, mx, my, pA[0], pA[1], metric=1 )
       
    a = r * math.tan( math.radians(angS) )
    c = math.sqrt(a**2 + r**2)
    
    newPoints = []    
    for i in range(int(subs/2)):
        if bowR: angA -= angS 
        else: angA += angS
        if angA < 0: angA += 360         
        if angA >= 360: angA -=360
        p = circlePoint( self, mx, my, angA, c )
        newPoints.append( [p[0],p[1],r] )            
        if not i: angS *= 2

    #print("a",a)
    #print("c",c)
    #print("newP",newPoints)
    #print("alpha",alpha)
    #print("r",r)
    #print("m", (mx,my) )
    #print("dis",dis)
    
    return newPoints# [["x","y","r"],...]
    


#############################################################################
#                                                                           #
#    20.00 point is on line-segment                                         #
#                                                                           #
#############################################################################
def pointOnLine( self, lPt1, lPt2, pt3):

    #---define points---
    x1,y1 = lPt1
    x2,y2 = lPt2
    x3,y3 = pt3

    #---Check If Point 3 Is On The Line---
    crossproduct = ((y3-y1) * (x2-x1)) - ((x3-x1) * (y2-y1))
    if math.fabs(crossproduct) > 0: return False

    #---check if point 3 is between point 1 and 2---
    pt3_between = (min(x1,x2) <= x3 <= max(x1,x2)) and (min(y1,y2) <= y3 <= max(y1,y2))          
    
    return pt3_between



#############################################################################
#                                                                           #
#    21.00 Rotate Point                                                     #
#                                                                           #
#############################################################################
def rotatePoint( self, x, y, angle):
    
    xNew = (  x * math.cos(math.radians(angle)) ) + ( y * math.sin(math.radians(angle)) )   
    yNew = ( -x * math.sin(math.radians(angle)) ) + ( y * math.cos(math.radians(angle)) )   

    return [xNew,yNew]

