'''
#################################################################################
#                                                                               #
#    Convert Library                                                            #
#                                                                               #
#################################################################################
# Copyright 2022 Kai Masemann                                                   #
#################################################################################
# This file is part of Lonnox CUT                                               #
#                                                                               #
#   Lonnox CUT is free software: you can redistribute it and/or modify          #
#   it under the terms of the GNU General Public License as published by        #
#   the Free Software Foundation, either version 3 of the License, or           #
#   (at your option) any later version.                                         #
#                                                                               #
#   Lonnox CUT is distributed in the hope that it will be useful,               #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of              #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
#   GNU General Public License for more details.                                #
#                                                                               #
#   You should have received a copy of the GNU General Public License           #
#   along with Lonnox CUT.  If not, see <http://www.gnu.org/licenses/>.         #
#                                                                               #
#################################################################################
#	C O N T E N T	     					                                    #
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 Load Librarys                                                        #
#                                                                               #
#    01.00 Global Variables                                                     #
#                                                                               #
#    02.00 DXF Read                                                             #  
#                                                                               #
#    03.00 DXF Find Block Content                                               #  
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 Load Libraries                                                       #
#                                                                               #
#################################################################################

#---Libraries For The Layout---
import math
import os
import sys
import uni
#import gcode
#import subprocess
from copy import copy, deepcopy


#################################################################################
#                                                                               #
#    01.00 Gloabl Variables                                                     #
#                                                                               #
#################################################################################

#---File And Dirnames for frozen/ unfrozen application---
if getattr(sys, 'frozen', False):
    # frozen
    path = os.path.dirname(sys.executable) + "/"
else:
    # unfrozen
    path = os.path.dirname(os.path.realpath(__file__)) + "/"
homePath = os.path.expanduser("~") + "/LonnoxCUT/"



#############################################################################
#                                                                           #
#    02.00 DXF Read                                                         #
#                                                                           #
#############################################################################
# Entities:
#   Are used to describe graphical Objects, Lonnox CUT needs only the Entities
#   thas describes geometrical forms.
# Blocks:
#   The DXF Format allows geometrical forms to be grouped in BLOCKS. These 
#   blocks can than be place multiple times, with different position, scale
#   and rotation values.
#   Blocks can also be nested (A Block contain a block (via INSERT), that 
#   contain another...).
# Splines:
#   R2012 dxf always contains interpolatet splines as polyline entity.
#   R2018 dxf contain polyline, lwpolyline or spline entities. Always given
#   are the control points of a spline, fitpoints are optional.
#   Instead of Start and Endtangent Autocad add an additional control point
#   to the spline.
############################################################################# 
def read( filepath ):
    
    #---Read DXF File---
    entys = [] 
    with open( filepath ) as file: 
        rows = file.read().split("\n")

        #---Search For Entity Content----------------------------------------
        e = "" #entity name
        g = "" #vertex group
        blkPos = [0,0,0] #Blockposition [x,y,z]
        blkName = "" #Blockname        
        for i in range(len(rows)-1):
            if i%2: continue 
            dxfCode = rows[i].strip()
            dxfValue = rows[i+1].strip()
            
            #---And New Entity---
            if dxfCode=="0":
                e = ""
                #---Filter Useful Entities---
                filter = ["ARC","CIRCLE","ELLIPSE","LINE","POINT","LWPOLYLINE","POLYLINE",
                          "SPLINE","SEQEND","BLOCK","ENDBLK","INSERT","VERTEX"]
                if dxfValue in filter:
                    e = dxfValue
                    #---Create Entity With Default Values---
                    #Added later if Entity Is Part Blockcontent: "blkName":"","blkXyz":[0,0,0],
                    if   e=="ARC"       : entys.append({"entity":e,"layer":"","color":"0",
                                                        "xyzr":[0,0,0,0],"angleRange":[0,360]})
                    elif e=="CIRCLE"    : entys.append({"entity":e,"layer":"","color":"0",
                                                        "xyzr":[0,0,0,0]})
                    elif e=="ELLIPSE"   : entys.append({"entity":e,"layer":"","color":"0",
                                                        "xyzCenter":[0,0,0],"xyzMajor":[0,0,0],
                                                        "ratio":1,"piRange":[0,6.28]})
                    elif e=="LINE"      : entys.append({"entity":e,"layer":"","color":"0",
                                                        "xyzStart":[0,0,0],"xyzEnd":[0,0,0]})
                    elif e=="POINT"     : entys.append({"entity":e,"layer":"","color":"0",
                                                        "xyz":[0,0,0]})
                    elif e=="LWPOLYLINE": entys.append({"entity":"POLYLINE","layer":"","color":"0",
                                                        "closed":False,"curve":"Polyline",
                                                        "smooth":"0","xyzbc":[]})
                    elif e=="POLYLINE"  : entys.append({"entity":e,"layer":"","color":"0",
                                                        "closed":False,"curve":"Polyline",
                                                        "smooth":"0","xyzbc":[]})
                    elif e=="SPLINE"    : entys.append({"entity":e,"layer":"","color":"0",
                                                        "closed":False,"degree":2,
                                                        "xyzwControl":[],"xyzFit":[]})
                    elif e=="INSERT"    : entys.append({"entity":e,"layer":"","color":"0",
                                                        "content":"","xyz":[0,0,0],
                                                        "scale":[1,1,1],"rotation":0})
            #---Add ARC---
            elif e=="ARC":
                #"{entity":,"layer":,"color":,"xyzr":[],"angleRange":[]}
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="10": entys[-1]["xyzr"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="20": entys[-1]["xyzr"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="30": entys[-1]["xyzr"][2] = float(dxfValue) - blkPos[2]
                elif dxfCode=="40": entys[-1]["xyzr"][3] = float(dxfValue)
                elif dxfCode=="50": entys[-1]["angleRange"][0] = float(dxfValue)
                elif dxfCode=="51": entys[-1]["angleRange"][1] = float(dxfValue)
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add CIRCLE---
            elif e=="CIRCLE":
                #{"entity":,"layer":,"color":,"xyzr":[]}
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="10": entys[-1]["xyzr"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="20": entys[-1]["xyzr"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="30": entys[-1]["xyzr"][2] = float(dxfValue) - blkPos[2]
                elif dxfCode=="40": entys[-1]["xyzr"][3] = float(dxfValue)
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add ELLIPSE---
            elif e=="ELLIPSE":
                #{"entity":,"layer":,"color":,"xyzCenter":[],"xyzMajor":[],"ratio":,"piRange":[]}
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="10": entys[-1]["xyzCenter"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="20": entys[-1]["xyzCenter"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="30": entys[-1]["xyzCenter"][2] = float(dxfValue) - blkPos[2]
                elif dxfCode=="11": entys[-1]["xyzMajor"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="21": entys[-1]["xyzMajor"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="31": entys[-1]["xyzMajor"][2] = float(dxfValue) - blkPos[2]
                elif dxfCode=="40": entys[-1]["ratio"] = float(dxfValue)
                elif dxfCode=="41": entys[-1]["piRange"][0] = float(dxfValue)
                elif dxfCode=="42": entys[-1]["piRange"][1] = float(dxfValue)
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add LINE---
            elif e=="LINE":
                #{"entity":,"layer":,"color":,"xyzStart":[],"xyzEnd":[]}
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="10": entys[-1]["xyzStart"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="20": entys[-1]["xyzStart"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="30": entys[-1]["xyzStart"][2] = float(dxfValue) - blkPos[2]
                elif dxfCode=="11": entys[-1]["xyzEnd"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="21": entys[-1]["xyzEnd"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="31": entys[-1]["xyzEnd"][2] = float(dxfValue) - blkPos[2]
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add POINT---
            elif e=="POINT":
                #entys.append({"entity":,"layer":,"color":,"xyz":[]})
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="10": entys[-1]["xyz"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="20": entys[-1]["xyz"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="30": entys[-1]["xyz"][2] = float(dxfValue) - blkPos[2]
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add LWPOLYLINE---
            elif e=="LWPOLYLINE":
                #{"entity":,"layer":,"color":,"closed":,"smooth":,"xyzbc":[x,y,z,bulge,isControlPoint]}
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="70": entys[-1]["closed"] = int(dxfValue)&1 != 0
                elif dxfCode=="10": entys[-1]["xyzbc"].append( [float(dxfValue)-blkPos[0],0,0,0,False] ) 
                elif dxfCode=="20": entys[-1]["xyzbc"][-1][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="42": entys[-1]["xyzbc"][-1][3] = float(dxfValue) #optional
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add SPLINE---
            elif e=="SPLINE":
                #{"entity":,"layer":,"color":,"closed":,"type":,"xyzwControl":[x,y,z,weight]}
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="70": entys[-1]["closed"] = int(dxfValue)&1 != 0
                elif dxfCode=="10": entys[-1]["xyzwControl"].append( [float(dxfValue)-blkPos[0],0,0,0] )
                elif dxfCode=="20": entys[-1]["xyzwControl"][-1][1] = float(dxfValue) - blkPos[1] 
                elif dxfCode=="30": entys[-1]["xyzwControl"][-1][2] = float(dxfValue) - blkPos[2] #optional
                elif dxfCode=="41": entys[-1]["xyzwControl"][-1][3] = float(dxfValue) #optional
                elif dxfCode=="11": entys[-1]["xyzFit"].append( [float(dxfValue)-blkPos[0],0,0] )
                elif dxfCode=="21": entys[-1]["xyzFit"][-1][1] = float(dxfValue) - blkPos[1] 
                elif dxfCode=="31": entys[-1]["xyzFit"][-1][2] = float(dxfValue) - blkPos[2] #optional
                elif dxfCode=="71": entys[-1]["degree"] = int(dxfValue) #optional 2=quadratic,3=cubic
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add POLYLINE Basics---
            elif e=="POLYLINE":
                #{"entity":,"layer":,"color":,"closed":,"smooth":,"xyzbc":[x,y,z,bulge,isControlPoint]}
                g = "POLYLINE"
                if   dxfCode== "8": entys[-1]["layer"] = dxfValue
                elif dxfCode=="62": entys[-1]["color"] = dxfValue
                elif dxfCode=="70": 
                    if int(dxfValue)&1: entys[-1]["closed"] = True
                    if int(dxfValue)&2: entys[-1]["curve"] = "Curve-fit"
                    if int(dxfValue)&4: entys[-1]["curve"] = "Spline-fit"
                elif dxfCode=="75":                 #optional 0=No, 5=Quadratic, 6=Cubic, 8=Bezier
                    if   dxfValue == "5": entys[-1]["smooth"] = "Quadratic" 
                    elif dxfValue == "6": entys[-1]["smooth"] = "Cubic" 
                    elif dxfValue == "8": entys[-1]["smooth"] = "Bezier" 
                    else                : entys[-1]["smooth"] = "No" 
                if blkName != "" and not "blockkName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Add POLYLINE VERTECIES---
            elif e=="VERTEX" and g=="POLYLINE":
                if   dxfCode=="10": entys[-1]["xyzbc"].append( [float(dxfValue)-blkPos[0],0,0,0,False] ) 
                elif dxfCode=="20": entys[-1]["xyzbc"][-1][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="30": entys[-1]["xyzbc"][-1][2] = float(dxfValue) - blkPos[2] #optional
                elif dxfCode=="42": entys[-1]["xyzbc"][-1][3] = float(dxfValue) #optional
                elif dxfCode=="70": entys[-1]["xyzbc"][-1][4] = int(dxfValue)&16 != 0  
            #---End POLYLINE---
            elif e=="SEQEND": g = ""
            #---Add BLOCK Content---
            #{"entity":,"layer":,"color":,"content":,"xyz":[],"scale":[],"rotation":}
            elif e=="INSERT":
                if   dxfCode== "2": entys[-1]["content"] = dxfValue
                elif dxfCode=="10": entys[-1]["xyz"][0] = float(dxfValue) - blkPos[0]
                elif dxfCode=="20": entys[-1]["xyz"][1] = float(dxfValue) - blkPos[1]
                elif dxfCode=="30": entys[-1]["xyz"][2] = float(dxfValue) - blkPos[2]
                elif dxfCode=="41": entys[-1]["scale"][0] = float(dxfValue)
                elif dxfCode=="42": entys[-1]["scale"][1] = float(dxfValue)
                elif dxfCode=="43": entys[-1]["scale"][2] = float(dxfValue)
                elif dxfCode=="50": entys[-1]["rotation"] = float(dxfValue)
                if blkName != "" and not "blockName" in entys[-1].keys(): 
                    entys[-1].update( {"blockName":blkName} )
            #---Store BLOCK Basics---
            elif e=="BLOCK":
                if   dxfCode== "3": blkName = dxfValue
                elif dxfCode=="10": blkPos[0] = float(dxfValue) 
                elif dxfCode=="20": blkPos[1] = float(dxfValue)
                elif dxfCode=="30": blkPos[2] = float(dxfValue)
            #---End BLOCK---
            elif e=="ENDBLK": blkPos = [0,0,0]; blkName = ""

        #print("---Dxf Raw------------------------------------------") 
        #for s in entys:
        #    print("269 dxf",s)
        
        #---Replace Blocks By Its Content------------------------------------
        i = 0 
        entys2 = []
        for i in range(len(entys)):
            #---Search And Add Block Content To Block Free List---
            if entys[i]["entity"] == "INSERT" and not "blockName" in entys[i].keys():
                blockEntys = blockContent(entys, entys[i])
                for enty in blockEntys:
                    entys2.append(enty)
            #---Add Entities That Are No Block Content To Block Free List---
            elif entys[i]["entity"] != "INSERT" and not "blockName" in entys[i].keys():
                entys2.append(entys[i])            
            
        #---Cleanup Block Names---
        for enty in entys2:
            if "blockName" in enty.keys(): del enty["blockName"]
         
        print("---Dxf Entities------------------------------------------") 
        try: print("Datei: ",filepath.split("/")[-1]) 
        except: pass
        for e in entys2:
            print("dxf.py 309 | ",e)
        
    return entys2    



#############################################################################
#                                                                           #
#    02.10 DXF Find Block Content                                           #
#                                                                           #
#############################################################################
# Block Example Structure:
#      BLOCK U2                             --> Block xyz speichern
#        LINE XYZ1 XYZ2                     --> U2 + LINE to Entys
#        CIRCLE XYZ RADIUS                  --> U2 + CIRCLE to Entys
#      ENDBLK
#      BLOCK U3 XYZ                         --> Block xyz speichern
#        INSERT U2 XYZ                      --> U3 + INSERT(U2,XYZ,SCALE,ROT) to Entys
#        INSERT U2 XYZ SCALE ROTATION(120)  --> U3 + INSERT(U2,XYZ,SCALE,ROT) to Entys
#        INSERT U2 XYZ SCALE ROTATION(240)  --> U3 + INSERT(U2,XYZ,SCALE,ROT) to Entys
#      ENDBLK
#      ENTITIES
#        INSERT U3 XYZ                      --> INSERT(U3,XYZ,SCALE,ROT) to Entys
############################################################################# 
def blockContent( entys, insertEnty ):
    
    #---Set Orientation Data---
    scale = insertEnty["scale"]
    rotation = insertEnty["rotation"]
    xyz = insertEnty["xyz"]

    #---Search For Block Content---
    blockEntys = []
    for i in range(len(entys)):
        j = len(entys)-1-i
        #---Search For Blockname---
        if not "blockName" in entys[j].keys(): continue
        if insertEnty["content"] == entys[j]["blockName"]:
            #---If The Found Block Entity Is Another Block---
            if entys[j]["entity"] == "INSERT":
                for enty in blockContent(entys,entys[j]):
                    bEnty = blockOrientation(deepcopy(enty),scale,rotation,xyz)                    
                    blockEntys.insert(0,bEnty) 
            #---If The Found Block Entity Is A Geometrie Add It---
            else:
                bEnty = blockOrientation(deepcopy(entys[j]),scale,rotation,xyz)                    
                blockEntys.insert(0,bEnty)
    
    return blockEntys



#############################################################################
#                                                                           #
#    02.11 DXF Orientation Of Block Content                                 #
#                                                                           #
#############################################################################
#Every block contains a base point. The origin of a block is the value X0,Y0, 
#Z0 of its contents. The base point can shift this origin.
#A base point with a positive x value will shift the content of the block 
#in the negative x direction. The base point is 0.0 in most cases. Because 
#the base point have impact of the whole block content it is substracted 
#directly in read() method as blkPos.
#If a block contains INSERT entities, there are also a insert position, 
#a scale and a rotation value. The have to be calculated in a fixed order.
#First the content of the INSERT must be scaled, second it must be rotated
#around its origin. Third the content is placed at the insert position
#(insertPosition + contentPosition).
#############################################################################
def blockOrientation( entity,insScale,insRotate,insXyz ):

    #---List Relevant Keys Of Points---
    keys = []
    if   entity["entity"] == "ARC"       : keys = ["xyzr"]  
    elif entity["entity"] == "CIRCLE"    : keys = ["xyzr"]  
    elif entity["entity"] == "ELLIPSE"   : keys = ["xyzCenter","xyzMajor"]     
    elif entity["entity"] == "LINE"      : keys = ["xyzStart","xyzEnd"]     
    elif entity["entity"] == "LWPOLYLINE": keys = ["xyb"]   
    elif entity["entity"] == "SPLINE"    : keys = ["xyz"]   
    elif entity["entity"] == "POLYLINE"  : keys = ["xyzbc"]   

    #---Scale, Rotate, Position Relevant Points---
    for key in keys: 
        #---Single Points(ARC,CIRCLE,ELLIPSE,LINE)---
        if not isinstance(entity[key][0],list):
            for i in range(3): 
                entity[key][i] *= insScale[i]
                entity[key][i] = round(entity[key][i],1)#########################
            entity[key][:2] = pRotate(entity[key][:2],insRotate)  
            for i in range(3): 
                entity[key][i] += insXyz[i]
                entity[key][i] = round(entity[key][i],1)#########################
        #---List Of Points(LWPOLYLINE,SPLINE,POLYLINE)---
        else:
            if entity["entity"] == "LWPOLYLINE": r = 2
            else: r = 3
            for j in range(len(entity[key])):
                for i in range(r): 
                    entity[key][j][i] *= insScale[i] 
                    entity[key][j][i] = round(entity[key][j][i],1)#########################
                entity[key][j][:2] = pRotate(entity[key][j][:2],insRotate)  
                for i in range(r): 
                    entity[key][j][i] += insXyz[i] 
                    entity[key][j][i] = round(entity[key][j][i],1)#########################
    
    return entity
    
    

#############################################################################
#                                                                           #
#    02.12 DXF 2D Point Rotation                                            #
#                                                                           #
#############################################################################
def pRotate(point, degree):

    #---Rotate Point---
    angle = math.radians(degree)
    qx = math.cos(angle) * point[0] - math.sin(angle) * point[1]
    qy = math.sin(angle) * point[0] + math.cos(angle) * point[1]
    
    return qx, qy 

    
###End    