#!/usr/bin/python3
# coding=utf-8

'''
#################################################################################
#                                                                               #
#    Lonnox CUT - G-Code Generator                                              #
#                                                                               #
#################################################################################
# Copyright 2022 Kai Masemann                                                   #
#################################################################################
# This file is part of Lonnox CUT                                               #
#                                                                               #
#   Lonnox CUT is free software: you can redistribute it and/or modify          #
#   it under the terms of the GNU General Public License as published by        #
#   the Free Software Foundation, either version 3 of the License, or           #
#   (at your option) any later version.                                         #
#                                                                               #
#   Lonnox CUT is distributed in the hope that it will be useful,               #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of              #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
#   GNU General Public License for more details.                                #
#                                                                               #
#   You should have received a copy of the GNU General Public License           #
#   along with Lonnox CUT.  If not, see <http://www.gnu.org/licenses/>.         #
#                                                                               #
#################################################################################
#    C O N T E N T S                                                            #
#-------------------------------------------------------------------------------#
#    01.00.00 global variables                                                  #
#                                                                               #
#    02.00.00 Lonnox CUT Window Class                                           #
#       01.00 Init Window Class                                                 #
#       02.00 Key Press Events                                                  #
#       03.00 Close Events                                                      #
#       04.00 Show About Screen                                                 #
#       05.00 Resize Event                                                      #
#       06.00 Move Main Window To Available Area                                #
#       07.00 Open Config Filder                                                #
#       08.00 Show Job Settings                                                 #
#                                                                               #
#    03.00.00 Modified QGraphicsView Class                                      #
#       03.01.00 Override Wheele Event                                          #
#                                                                               #
#    04.00.00 Lonnox CUT User Interface Class                                   #
#       01.00 Init Widget Class                                                 #
#       02.00 Create Basic Layout                                               #
#          01 Insert Preview Section (Upper Area)                               #
#          02 Insert Module Section (Left Column)                               #
#          03 Insert Edit Section (Middle Column)                               #
#          04 insert job section (Right Column)                                 #
#       03.00 Enable Code View                                                  #
#       04.00 Enable Paint View                                                 #
#       05.00 Enable Module View                                                #
#       06.00 Show Module Information                                           #
#       07.00                                                                   #
#       08.00                                                                   #
#       09.00 Choose File Parameter                                             #
#       10.00 Add Axis Cross To Draw Preview                                    #
#       11.00 Confiurate The Draw Preview                                       #
#       12.00 Show Module Status                                                #
#                                                                               #
#    05.00.00 Lonnox CUT/Job Settings                                           #
#       01.00 Init Application Class                                            #
#       02.00 Create Basic Layout                                               #
#          01 Window Setup                                                      #
#          02 Create Layout                                                     #
#       03.00 Create Spin Setting                                               #
#       04.00 Create Combo Setting                                              #
#                                                                               #
#    06.00.00 Lonnox CUT Application                                            #
#       01.00 Init Application Class                                            #
#       02.00 Load A Module From Module/Job Tree To Edit Area                   #
#         .01 Reset Edit Area                                                   #
#         .02 Load Modulename And Set Picture                                   #
#         .03 Load Modulelabels And Options                                     #
#         .04 Load Parameter From Autofill Table                                #
#         .05 Load Parameter From Joblist                                       #
#         .06 Finish Moduleloading                                              #
#       03.00 Add Job                                                           #
#       04.00 Show Jobs                                                         #
#       05.00 Delete Job                                                        #
#       06.00 Move Job Up                                                       #  
#       07.00 Move Job Down                                                     #
#       08.00 Dublicate Jobs                                                    #
#       09.00 Mirror Jobs                                                       #
#       10.00 Edit Job                                                          #
#       11.00 Move Job                                                          #
#       12.00 Enable/Disable Job                                                #
#       40.00 Save Job Settings                                                 #
#       41.00 File Menu - New                                                   #
#       42.00 File Menu - Open                                                  #
#       43.00 File Menu - Save                                                  #
#       44.00 File Menu - Save As                                               #
#       45.00 File Menu - Save Gcode                                            #
#       46.00 File Menu - Programm beenden                                      #
#       47.00 Option Menu - AddModules                                          #
#       49.00 Menu - Ask For Save Changes                                       #
#       80.00 Write ini File                                                    #
#       81.00 Read ini File                                                     #
#       83.00 Update View                                                       #
#          01 Build GCode                                                       #
#          02 Paint Preview                                                     #
#       84.00 Draw GCode                                                        #
#       85.00 Load Modules                                                      #
#       86.00 Load Languages                                                    #
#       87.00 New Machine                                                       #
#       88.00 Create Toollist                                                   #
#       89.00 Menu Import TrunCAD (Masterwood) dxf                              #
#                                                                               #
#    07.00.00 Language Translation                                              #
#                                                                               #
#    08.00.00 Run Application                                                   #
#                                                                               #
#################################################################################
'''
__version__ = '4.4.4.2'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

'''
#################################################################################
#    00.00 load libraries                                                       #
#################################################################################
'''

#---libraries for the layout---
import sys
from PyQt5.QtWidgets import * 
from PyQt5.QtGui import * 
from PyQt5.QtCore import *
from random import randint
import subprocess
import uni
import math
import os
import shutil
import importlib
from importlib import reload
import csv
import zipfile
import gcode
import convert
from copy import copy, deepcopy
import time
import dxf



#################################################################################
#                                                                               # 
#    01.00.00 global variables                                                  #
#                                                                               # 
#################################################################################

#---App Name And Version---
appname = "Lonnox CUT v" + __version__

#---Job Data For Generating G Code---
# data structure: every sublist in the list represent one job
# elements in sublist: [[jobSetup],modulename,zeroPointX,zeroPointY,orientation,Parameter1,Parameter2,...] 
# elements in jobSetup: [Disable/Enable,Job einrücken,errMarks]
# elements in errMarks: [zeroPointX,zeroPointY,orientation,Parameter1,Parameter2,...] (not saved in cut Files) 
joblist = [] 

#---Name And Reference Of The Found Module, Languages, Machines--- 
modulefiles = {}
langfiles = {}
machines = []
machine = None
dxfFilter = None

#---Language File Contents---
L = None

#---list For Collecting Autofill Values---
autofill = [[],[]]

#---File And Dirnames---
if getattr(sys, 'frozen', False):
    # frozen
    path = os.path.dirname(sys.executable) + "/"
else:
    # unfrozen
    path = os.path.dirname(os.path.realpath(__file__)) + "/"
    #path = os.getcwd() + "/"
homePath = os.path.expanduser("~") + "/LonnoxCUT/"
jobfile = ""
jobdir = ""
gcodedir = ""
importDir = ""
importFile = ""

#---Application Settings---
wxpos = 0
wypos = 0
win = None

#---Reference For GUI Widgets---
app = None
mGui = None
sGui = None

#---Event Handling---
pEvent = 1 #preview

#---Debug On Off---
debugMode = 0


#################################################################################
#                                                                               # 
#    02.00.00 CUT Window Class                                                 #
#                                                                               # 
#################################################################################
class mainWindow( QMainWindow ):
    '''
    #for creating menu-, satus- ,toolbars in the UI this QMainWindow class is
    #needed by Qt. 
    '''

    #############################################################################
    #    02.01.00 Init Window Class                                             #
    #############################################################################
    def __init__( self ):
        global wxpos
        global wypos
        global mGui
        global path
        global appname
        global machine, machines
        
        
        self.key = ""
        self.viewScale = 1
        
        #---Setup Qt Main Window--- 
        QMainWindow.__init__(self)
        self.qdw = QDesktopWidget()
        self.moveToAvailable()
        self.qdw.screenCountChanged.connect(self.moveToAvailable)
        
        self.setWindowIcon( QIcon(path + "data/icon.png") )
        self.setWindowTitle(appname)

        #---Load Widgets---
        mGui = mainGui()
        self.setCentralWidget(mGui)

        #---Menu Bar Actions---
        actNew = QAction( LT("New"), self)
        actNew.setStatusTip( LT("Delete joblist") )
        actNew.triggered.connect(app.menuNew)

        actOpen = QAction( LT("Open file"), self)
        actOpen.setStatusTip( LT("Open joblist file") )
        actOpen.triggered.connect(app.menuOpen)

        actSave = QAction( LT("Save"), self)
        actSave.setStatusTip( LT("Save joblist file") )
        actSave.setShortcut("Ctrl+S")
        actSave.triggered.connect(app.menuSave)

        actSaveAs = QAction( LT("Save as..."), self)
        actSaveAs.setStatusTip( LT("Save joblist with new name") )
        actSaveAs.triggered.connect(app.menuSaveAs)

        actSaveGcode = QAction( LT("Save gcode file"), self)
        actSaveGcode.setStatusTip( LT("Generate and save gcode from the joblist") )
        actSaveGcode.setShortcut("Ctrl+G")
        actSaveGcode.triggered.connect(app.menuSaveGcode)

        actDxfImport = QAction( LT("Import dxf file..."), self)
        actDxfImport.setStatusTip( LT("Import dxf file...") )
        actDxfImport.triggered.connect(app.menuDxfImport)

        actExit = QAction( LT("Quit"), self)
        actExit.setStatusTip( LT("Save workspace and quit Lonnox CUT") )
        actExit.triggered.connect(app.menuExit)
        actExit.MenuRole(QAction.NoRole) #Disable Mac Menu Move

        actAddModules = QAction( LT("Add modules..."), self)
        actAddModules.setStatusTip( LT("Add new modules from zip file to Lonnox CUT") )
        actAddModules.triggered.connect(app.menuAddModules)

        if uni.g0Mode: self.actG0 = QAction( QIcon(path + "data/Yes.png"),LT("Show/Hide G0 Path"), self)
        else: self.actG0 = QAction( QIcon(path + "data/No.png"),LT("Show/Hide G0 Path"), self)
        self.actG0.setStatusTip( LT("Show/Hide G0 Path") )
        self.actG0.triggered.connect(self.setG0Mode)

        self.actPref = QAction( LT("Settings"), self)
        self.actPref.setStatusTip( LT("Lonnox CUT and joblist settings") )
        self.actPref.triggered.connect(self.jobSettings)
        self.actPref.MenuRole(QAction.NoRole) #Disable Mac Menu Move
        
        actConfig = QAction( LT("Configfolder"), self)
        actConfig.setStatusTip( LT("Open the folder with config data") )
        actConfig.triggered.connect(self.openConfig)

        actHelp = QAction( LT("About Lonnox CUT"), self)
        actHelp.setStatusTip( LT("About Lonnox CUT and licence") )
        actHelp.triggered.connect(self.aboutScreen)
        actHelp.MenuRole(QAction.NoRole) #Disable Mac Menu Move

        #---Create File Menu---
        self.fileMenu = self.menuBar().addMenu( LT("File") )
        self.fileMenu.addAction(actNew)         
        self.fileMenu.addAction(actOpen)         
        self.fileMenu.addAction(actDxfImport)         
        self.fileMenu.addSeparator()
        self.fileMenu.addAction(actSave)         
        self.fileMenu.addAction(actSaveAs)         
        self.fileMenu.addAction(actSaveGcode)         
        self.fileMenu.addSeparator()
        self.fileMenu.addAction(actExit)         

        #---Create Option Menu---
        self.optMenu = self.menuBar().addMenu( LT("Options") )
        self.optMenu.addAction(actAddModules)
        self.optMenu.addAction(self.actG0)
        self.optMenu.addAction(self.actPref)
        self.optMenu.addAction(actConfig)
        
        #---Create Help Menu--- 
        self.helpMenu = self.menuBar().addMenu( LT("Help") )
        self.helpMenu.addAction(actHelp)         

        #---Create Machine ComboBoxes---
        self.machineCombo = QComboBox(self)
        self.machineCombo.setFont( QFont( "Arial",10) )
        self.machineCombo.currentIndexChanged.connect(app.newMachine)
        self.machineCombo.addItems(machines)
        try: self.machineCombo.setCurrentIndex( machines.index( machine ) )
        except: pass
        if sys.platform == "darwin": #On Mac platforms
            self.machineCombo.resize(200,23)
            self.machineCombo.move(950-200,0)
        else: self.menuBar().setCornerWidget( self.machineCombo )

        #---Create Status Bar---  
        self.statusBar() 

        #---Add Timer Event For External Events----
        self.timer = QTimer(self)
        self.timer.timeout.connect(mGui.moduleStatus)
        self.timer.start(100)        



    #############################################################################
    #    02.02.00 Key Press Events                                              #
    #############################################################################
    def keyPressEvent( self, event ):
        if event.key() == Qt.Key_Escape:
            self.key = "ESC"
            uni.key = "ESC"


    #############################################################################
    #    02.03.00 Close Events                                                  #
    #############################################################################
    def closeEvent( self, event ):
        self.timer.stop()
        app.menuExit()


    #############################################################################
    #    02.04.00 Show About Screen                                             #
    #############################################################################
    def aboutScreen( self ):
        global path                          

        #---Show Message---
        about = QMessageBox( self )
        about.setIconPixmap( QPixmap( path + "data/license.png") )
        about.setWindowTitle( LT("About Lonnox CUT") )
        about.show()


    #############################################################################
    #    02.05.00 Resize Events                                                 #
    #############################################################################
    def resizeEvent( self, event ):
        global mGui

        sw = 950                    #standard width
        sh = 778                    #standard height
        w = self.width()            #window width 
        h = self.height()           #window height
        wf = w / sw                 #widthFactor

        #---Resize Preview Area---
        mGui.paintView.resize(w,h-(sh-300) )
        mGui.moduleView.resize(w,h-(sh-300) )
        mGui.codeView.resize(w,h-(sh-300) )
        mGui.viewMode.resize(w,h-(sh-300) )
        mGui.partInfoLbl.resize(int(950*wf),20)
        mGui.partInfoLbl.move(int(5*wf),2)
        mGui.moduleBtn.resize(int(60*wf),30)
        mGui.moduleBtn.move(int(388*wf),h-(sh-300)-30)
        mGui.paintBtn.resize(int(60*wf),30)
        mGui.paintBtn.move(int(445*wf),h-(sh-300)-30)
        mGui.codeBtn.resize(int(60*wf),30)
        mGui.codeBtn.move(int(502*wf),h-(sh-300)-30)
        mGui.coordsLbl.resize(int(200*wf),20)
        mGui.coordsLbl.move(int(0*wf),h-(sh-280))
        mGui.zoomInBtn.resize(int(30*wf),30)
        mGui.zoomInBtn.move(int(920*wf),h-(sh-240))
        mGui.zoomOutBtn.resize(int(30*wf),30)
        mGui.zoomOutBtn.move(int(920*wf),h-(sh-270))

        #---Resize ModuleTree Area---
        mGui.moduleTree.resize(int(280*wf), 425) 
        mGui.moduleTree.move(int(5*wf) , h-(sh-305) ) 

        #---Resize Edit Area---
        mGui.editLbl.resize(int(370*wf), 25)
        mGui.editLbl.move(int(290*wf), h-(sh-305) )
        mGui.toolLbl.resize(int(100*wf),25)
        mGui.toolLbl.move(int(560*wf), h-(sh-305) )
        y = 332
        for i in range( len(mGui.paramLbl) ):
            if i == 2:
                mGui.paramTitle.resize(int(370*wf),4)
                mGui.paramTitle.move(int(290*wf), h-(sh-(y+4)) )
                y += 10
            mGui.paramLbl[i].resize(int(201*wf),17)
            mGui.paramLbl[i].move(int(290*wf), h-(sh-(y+1)) )
            mGui.paramMode[i].resize(int(170*wf),19)
            mGui.paramMode[i].move(int(490*wf), h-(sh-y) )
            y += 18
        mGui.infoBtn.resize(int(30*wf),25)
        mGui.infoBtn.move(int(290*wf), h-(sh-705) )
        mGui.addBtn.resize(int(340*wf),25)
        mGui.addBtn.move(int(320*wf), h-(sh-705) )

        #---Resize JobTree Area---
        count = len(mGui.jobBtn)
        for i in range( count ):
            mGui.jobBtn[i].resize( int((280/count)*wf),25)
            mGui.jobBtn[i].move( int((665+(i*(280/count)))*wf), h-(sh-305) )
        mGui.jobTree.resize(int(280*wf), 400) 
        mGui.jobTree.move(int(665*wf) , h-(sh-330) ) 
        
        #---Resize Machine Option On Mac---
        if sys.platform == "darwin": 
            self.machineCombo.move(w-200,0)


    #############################################################################
    #    02.06.00 Move Main Window To Available Area                            #
    #############################################################################
    def moveToAvailable( self ):
        global wxpos, wypos
        
        #---Move Application Window To Available Screen Area---
        ag = self.qdw.availableGeometry()
        if ((wxpos+8+950) > ag.width()) or ((wypos+31+778) > ag.height()):
            wxpos=0; wypos=0
        #self.setGeometry( wxpos+8, wypos+31, 1383, 778 ) #Totorial Size
        self.setGeometry( wxpos+8, wypos+31, 950, 778 )
        

    #############################################################################
    #    02.07.00 Open Config Folder                                            #
    #############################################################################
    def openConfig( self ):
        
        #---Detect Platform/ File Manager---
        if sys.platform in ["linux2","linux"] and os.path.exists("/usr/share/Thunar"):
            subprocess.call(["thunar", os.path.expanduser("~")+"/LonnoxCUT"])
        elif sys.platform == "win32":
            subprocess.call( ["explorer",os.path.expanduser("~")+"\LonnoxCUT"] )
        elif sys.platform == "darwin":
            subprocess.call( ["open",os.path.expanduser("~")+"/LonnoxCUT"] )
        else:
            #---Show Message---
            message = QMessageBox( self )
            message.setText( LT("Unknown platform/filemanager")+":\n"+homePath )
            message.setWindowTitle( " " )
            message.show()


    #############################################################################
    #    02.08.00 Show Job Settings                                             #
    #############################################################################
    def jobSettings( self ):
        global sGui

        #---Show Job Sertings---
        sGui = setGui()
        sGui.show()


    #############################################################################
    #    02.09.00 Show/Hide G0 Mode                                             #
    #############################################################################
    def setG0Mode( self ):
        global mGui
        
        #---Toggle G0 Mode---
        if uni.g0Mode:
            uni.g0Mode = 0
            self.actG0.setIcon( QIcon(path + "data/No.png") )
        else:
            uni.g0Mode = 1
            self.actG0.setIcon( QIcon(path + "data/Yes.png") )
        app.preview()


 
#################################################################################
#                                                                               # 
#    03.00.00 Modified QGraphicsView Class                                      #
#                                                                               # 
#################################################################################
zoomFactor = 1.1
mousePressed = False
class myGraphicsView( QGraphicsView ):
    '''
    #adopt methods from the original QtGui.QGraphicsView class and
    #overrides the mouse wheel event to ignore the scrollbar events
    #and replace it with zoom functions.
    '''
    
    #############################################################################
    #    03.01.00 Override Wheele Event                                         #
    #############################################################################
    def wheelEvent( self, event ):
        global win,zoomFactor
        
        #---Adapt Windows PyQt5/ Linux PyQt4---
        d = 1.0
        d = event.angleDelta().y()
        
        #---Scale With Mouse Wheel---
        if d > 0: 
            win.viewScale *= zoomFactor 
            self.scale(zoomFactor, zoomFactor)
        elif d < 0: 
            win.viewScale /= zoomFactor 
            self.scale(1/zoomFactor, 1/zoomFactor)
        app.preview( noCodeUpdate=1 )   
        

    #############################################################################
    #    03.02.00 Override Enter Event                                          #
    #############################################################################
    def enterEvent( self, event ):
        self.viewport().setCursor( QCursor( QPixmap( path + "data/cross.png" ) ) )
        

    #############################################################################
    #    03.03.00 Override Release Event                                        #
    #############################################################################
    def mouseReleaseEvent( self, event ):
        global mousePressed

        self.viewport().setCursor( QCursor( QPixmap( path + "data/cross.png" ) ) )
        mousePressed = False
        

    #############################################################################
    #    03.04.00 Override Press Event                                          #
    #############################################################################
    def mousePressEvent( self, event ):
        global mousePressed 
        
        mousePressed = True
        super(myGraphicsView, self).mousePressEvent(event)
     

    #############################################################################
    #    03.05.00 Override Move Event                                           #
    #############################################################################
    def mouseMoveEvent( self, event ):
        global mousePressed, mGui, win

        #---Track Mouse Position And Convert To Pixel to mm---
        scene = self.mapToScene(event.pos())
        x = uni.settings[1] - scene.x()
        y = scene.y() 
        mGui.coordsLbl.setText("X:{:.2f}mm   Y:{:.2f}mm".format(x,y))
        #---If Mouse Is Pressed Execute Original Event (Dragging)---
        if mousePressed:
            super(myGraphicsView, self).mouseMoveEvent(event)



#################################################################################
#                                                                               # 
#    04.00.00 CUT User Interface Class                                         #
#                                                                               # 
#################################################################################
class mainGui( QWidget ):
    '''
    #for creating widgets in the UI this QWidget class is needed by Qt. 
    '''


    #############################################################################
    #    04.01.00 Init Widget Class                                             #
    #############################################################################
    def __init__( self ):

        #---Setup QtWidgets--- 
        QWidget.__init__(self)
        self.layout()


    #############################################################################
    #    04.02.00 Create Basic Layout                                           #
    #############################################################################
    def layout( self ):
        global modulefiles
        global machines
        

        #########################################################################
        #    04.02.01 Insert Preview Section (Upper Area)                       #
        #########################################################################

        #---Create The Paint Preview---
        self.scene = QGraphicsScene()
        self.scene.setBackgroundBrush( QBrush(QColor("#808080")) ) 
        self.paintView = myGraphicsView(self.scene, self) 
        self.paintView.setMouseTracking(True)
        self.paintView.setDragMode( QGraphicsView.ScrollHandDrag )
        self.paintView.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)  
        self.paintView.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        
        #---Paint Working Area--- 
        wa = QPainterPath()
        wa.addRect(0,0,uni.settings[1],uni.settings[2])
        self.scene.addPath(wa, QPen(Qt.black, 3, Qt.DotLine))

        self.paintView.rotate(uni.settings[5])
        #self.paintView.resize(950,300)

        #---Create The Module Preview---
        self.moduleView = QLabel(self)
        self.pm = QPixmap( path + "data/LCUT.png" )
        self.moduleView.setPixmap( self.pm ) 
        self.moduleView.setAlignment(Qt.AlignCenter)
        self.moduleView.setStyleSheet("QLabel { background-color: #808080;}")
        #self.moduleView.resize(950,300)
        
        #---Create The code Preview---
        self.codeView = QTextEdit(self)
        self.codeView.setReadOnly(1)
        self.cFont = QFont( "Arial",10) 
        self.cFont.setFixedPitch(1)
        self.cFont.setBold(1)
        self.codeView.setFont(self.cFont)
        self.codeView.setStyleSheet("QTextEdit { background-color: #d0d0d0; color: #473348; }") 
        #self.codeView.resize(950,300)

        #---Insert The Paint, GCode And Module Preview In A StackedWidget---
        self.viewMode = QStackedWidget(self)
        self.viewMode.addWidget(self.moduleView)
        self.viewMode.addWidget(self.paintView)
        self.viewMode.addWidget(self.codeView)
        self.viewMode.setCurrentIndex(0)       
        #self.viewMode.resize(950,300)

        #---Part Info Label---
        self.partInfoLbl = QLabel("",self)
        self.partInfoLbl.setStyleSheet("QLabel { color: #ffffff; }")
        self.partInfoLbl.setFont( QFont( "Arial",10) )
        self.partInfoLbl.setVisible(False)
        
        #---Create Preview Toggle Buttons---
        self.moduleBtn = QPushButton(self)
        self.moduleBtn.setToolTip( LT("Shows module pictures") )
        self.moduleBtn.setIcon( QIcon(path + "data/moduleView.png") )
        self.moduleBtn.setIconSize(QSize(32,24))
        self.moduleBtn.pressed.connect(self.moduleViewEnable)
        #self.moduleBtn.resize(60,30)
        #self.moduleBtn.move(388,270)

        self.paintBtn = QPushButton(self)
        self.paintBtn.setToolTip( LT("Shows GCode drawings preview") )
        self.paintBtn.setIcon( QIcon(path + "data/paintView.png") )
        self.paintBtn.setIconSize(QSize(32,24))
        self.paintBtn.pressed.connect(self.paintViewEnable)
        #self.paintBtn.resize(60,30)
        #self.paintBtn.move(445,270)

        self.codeBtn = QPushButton(self)
        self.codeBtn.setToolTip( LT("Shows GCode text preview") )
        self.codeBtn.setIcon( QIcon(path + "data/codeView.png") )
        self.codeBtn.setIconSize(QSize(32,24))
        self.codeBtn.pressed.connect(self.codeViewEnable)
        #self.codeBtn.resize(60,30)
        #self.codeBtn.move(502,270)
        
        self.coordsLbl = QLabel("X: 0.00mm   Y: 0.00mm",self)
        self.coordsLbl.setFont( QFont( "Arial",10) )
        self.coordsLbl.setAlignment(Qt.AlignCenter)
        self.coordsLbl.setVisible(False)

        self.zoomInBtn = QPushButton(self)
        self.zoomInBtn.setIcon( QIcon(path + "data/zoomIn.png") )
        self.zoomInBtn.setIconSize(QSize(24,24))
        self.zoomInBtn.pressed.connect(self.zoomIn)
        self.zoomInBtn.setVisible(False)
        #self.zoomInBtn.resize(30,30)
        #self.zoomInBtn.move(920,240)

        self.zoomOutBtn = QPushButton(self)
        self.zoomOutBtn.setIcon( QIcon(path + "data/zoomOut.png") )
        self.zoomOutBtn.setIconSize(QSize(24,24))
        self.zoomOutBtn.pressed.connect(self.zoomOut)
        self.zoomOutBtn.setVisible(False)
        #self.zoomOutBtn.resize(30,30)
        #self.zoomOutBtn.move(920,270)
        
        #########################################################################
        #    04.02.02 Insert Module Section (Left Column)                        #
        #########################################################################
        
        #---Create The Module Tree---
        self.moduleTree = QTreeWidget(self)
        self.moduleTree.setColumnCount(1)
        self.moduleTree.setHeaderLabel( LT("Modules") )
        self.moduleTree.sortItems(0,Qt.AscendingOrder)
        self.moduleTree.setSortingEnabled(1) 
        self.moduleTree.itemDoubleClicked.connect(app.editModule)
        self.moduleTree.itemActivated.connect(app.editModule)
        self.moduleTree.resize(280,425)
        self.moduleTree.move(5,305)

        #---Insert Modules In The Module Tree---
        section = {} 
        for name, ref in sorted(modulefiles.items()):
            if not ref.section in section:
                section[ref.section] = QTreeWidgetItem(self.moduleTree)
                section[ref.section].setText(0, LT(ref.section) )
            QTreeWidgetItem(section[ref.section]).setText(0, LT(ref.name) )

        #########################################################################
        #    04.02.03 Insert Edit Section (Middle Column)                       #
        #########################################################################

        #---Create The Edit Label---
        self.editLbl = QLabel("", self)
        self.editLbl.setFont( QFont( "Arial",16) )
        self.editLbl.setStyleSheet("QLabel { background-color: #404040; color: #f0f0f0; }")
        #self.editLbl.resize(370,25)
        #self.editLbl.move(290,305)

        #---Create PrefTool Label---
        self.toolLbl = QLabel( LT("Tool"), self)
        self.toolLbl.setPixmap(QPixmap( path + "data/cutter.png" ) ) 
        self.toolLbl.setToolTip( LT("This module use a predefined Tool") )
        self.toolLbl.setAlignment(Qt.AlignCenter)
        self.toolLbl.hide()
        #self.toolLbl.resize(100,25)
        #self.toolLbl.move(560,305)
        
        #---Create Parameter Label and Values--- 
        self.paramMode = []
        self.paramLbl = []
        self.paramEmpty = []
        self.paramText = []
        self.paramSpin = []
        self.paramCombo = []
        self.paramFile = []; self.pFile = [] 
        self.paramCheck = []
        y = 332
        for i in range(20):
            #---Create Separate Parameter Area---
            if i == 2:
                self.paramTitle = QLabel(self) 
                self.paramTitle.setStyleSheet("QLabel { background-color: #505050; }")
                #self.paramTitle.resize(370,4)
                #self.paramTitle.move(290, y+4)
                y += 10
            
            #---Create Parameter Labels---
            self.paramLbl.append(QLabel(self))
            self.paramLbl[i].setStyleSheet("QLabel { background-color: #d0d0d0; }")
            self.paramLbl[i].setFont( QFont( "Arial",10) )
            #self.paramLbl[i].resize(201,17)
            #self.paramLbl[i].move(290, y+1)

            #---Create Parameter Empty (bypass the Win8/Qt4.8 TextLine Disable Bug)---
            self.paramEmpty.append(QLineEdit(self))
            self.paramEmpty[i].setEnabled(False)
            self.paramEmpty[i].setReadOnly(True)
            
            #---Create Parameter Text---
            self.paramText.append(QLineEdit(self))
            self.paramText[i].setFont( QFont( "Arial",10) )
            self.paramText[i].editingFinished.connect(app.preview)

            #---Create Prameter SpinBoxes---
            self.paramSpin.append(QLineEdit(self))
            self.paramSpin[i].setFont( QFont( "Arial",10) )
            self.paramSpin[i].setAlignment(Qt.AlignRight)
            self.paramSpin[i].editingFinished.connect(app.preview)
            #self.paramSpin[i].editingFinished.connect(self.calculationCheck)
            if i < 2: self.paramSpin[i].setValidator( QRegExpValidator( QRegExp("[-]{0,1}\d{0,}\.\d{0,}"),self) )
            
            #---Create Parameter ComboBoxes---
            self.paramCombo.append(QComboBox(self))
            self.paramCombo[i].setFont( QFont( "Arial",10) )
            self.paramCombo[i].currentIndexChanged.connect(app.preview)

            #---Create Parameter File Button---
            self.paramFile.append(QPushButton(self))
            self.paramFile[i].setToolTip( LT("Choose file") )
            self.paramFile[i].setIcon( QIcon(path + "data/page_text.png") )
            self.paramFile[i].id = i #setText( str(i) )
            self.paramFile[i].pressed.connect(self.openFileParam)
            self.pFile.append( [""] )

            #---Create Parameter Check Button---
            self.paramCheck.append(QCheckBox(self))
            self.paramCheck[i].stateChanged.connect(app.preview)

            #---Insert Spin And ComboBox In A StackedWidget---
            self.paramMode.append(QStackedWidget(self))
            self.paramMode[i].addWidget(self.paramEmpty[i])
            self.paramMode[i].addWidget(self.paramText[i])
            self.paramMode[i].addWidget(self.paramSpin[i])
            self.paramMode[i].addWidget(self.paramCombo[i])
            self.paramMode[i].addWidget(self.paramFile[i])
            self.paramMode[i].addWidget(self.paramCheck[i])
            self.paramMode[i].setEnabled(False)
            #self.paramMode[i].resize(170,19)
            #self.paramMode[i].move(490, y)
 
            y += 18
         
        #---Create The Info Button---
        self.infoBtn = QPushButton(self)
        self.infoBtn.setToolTip( LT("Details for selected module") )
        self.infoBtn.setIcon( QIcon(path + "data/icon_info.png") )
        self.infoBtn.setEnabled(False)
        self.infoBtn.pressed.connect(self.moduleInfo)
        #self.infoBtn.resize(30,25)
        #self.infoBtn.move(290,705)

        #---Create The Job Add Button---
        self.addBtn = QPushButton( LT("Add job to joblist -->"), self)
        self.addBtn.setEnabled(False)
        self.addBtn.pressed.connect(app.addJob )
        #self.addBtn.resize(340,25)
        #self.addBtn.move(320,705)


        #########################################################################
        #    04.02.04 insert job section (Right Column)                         #
        #########################################################################
	
        #---Create Edit Buttons For The Job Tree---
        self.jobBtn = []
        self.jobBtnTip = [LT("Load parameters of the selected job for editing"),
                          LT("Insert a copy of the selected jobs"),
                          LT("Move selected jobs on x axis"),
                          LT("Move selected jobs on y axis"),
                          LT("Mirror selected jobs"),
                          LT("Delete selected Jobs"),
                          LT("Move selected jobs up"),
                          LT("Move selected jobs down"),
                          LT("Enable/disable selected jobs")]
        self.jobBtnIco = ["data/page_edit.png",
                          "data/copy.png",
                          "data/icon_moveX.png",
                          "data/icon_moveY.png",
                          "data/icon_mirror.png",
                          "data/delete.png",
                          "data/arrow_up.png",
                          "data/arrow_down.png",
                          "data/icon_eye.png"]
        self.jobBtnEvt = [app.editJob,
                          app.jobDuplicate,
                          app.jobMoveX,
                          app.jobMoveY,
                          app.jobMirror,
                          app.jobDelete,
                          app.jobUp,
                          app.jobDown,
                          app.jobDisable]
         
        count = len(self.jobBtnEvt)
        for i in range( count ): 
            self.jobBtn.append( QPushButton(self) )
            self.jobBtn[i].setToolTip( self.jobBtnTip[i] )
            self.jobBtn[i].setIcon( QIcon(path + self.jobBtnIco[i]) )
            self.jobBtn[i].pressed.connect( self.jobBtnEvt[i] )
   

        #---Create The Job Tree---
        self.jobTree = QTreeWidget(self)
        self.jobTree.setColumnCount(4)
        self.jobTree.setHeaderLabels( [LT("Jobs"),"","",""] )
        self.jobTree.setColumnWidth(0,200)
        self.jobTree.setColumnWidth(1,20)
        self.jobTree.setColumnWidth(2,20)
        self.jobTree.setColumnWidth(3,20)
        self.jobTree.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.jobTree.itemDoubleClicked.connect(app.editJob)


    #############################################################################
    #    04.03.00 Enable Code View                                              #
    #############################################################################
    def codeViewEnable( self ):
        self.viewMode.setCurrentIndex(2)       
        app.preview( self )
        #---Disable Coords And Zoom Buttons---
        self.partInfoLbl.setVisible(False)
        self.coordsLbl.setVisible(False)
        self.zoomInBtn.setVisible(False)
        self.zoomOutBtn.setVisible(False)
        

    #############################################################################
    #    04.04.00 Enable Paint View                                             #
    #############################################################################
    def paintViewEnable( self ):
        global win
        
        self.paintView.fitInView( -200,-200,uni.settings[1]+400,
                                  uni.settings[2]+400, Qt.KeepAspectRatio )
        #self.paintView.fitInView( 0,0,950,300, Qt.KeepAspectRatio )
        self.paintView.centerOn( uni.settings[1]/2, uni.settings[2]/2 )
        self.paintView.scale( 0.8, 0.8 )
        self.viewMode.setCurrentIndex(1)       
        win.viewScale = 1
        app.preview( self )
        #---Enable Coords And Zoom Buttons---
        self.partInfoLbl.setVisible(True)
        self.coordsLbl.setVisible(True)
        self.zoomInBtn.setVisible(True)
        self.zoomOutBtn.setVisible(True)


    #############################################################################
    #    04.05.00 Enable Module View                                            #
    #############################################################################
    def moduleViewEnable( self ):
        self.viewMode.setCurrentIndex(0)       
        #---Disable Coords And Zoom Buttons---
        self.partInfoLbl.setVisible(False)
        self.coordsLbl.setVisible(False)
        self.zoomInBtn.setVisible(False)
        self.zoomOutBtn.setVisible(False)


    #############################################################################
    #    04.06.00 Show Module Information                                       #
    #############################################################################
    def moduleInfo( self ):
        global modulefiles
        
        #---Show Message---
        if self.editLbl.text() == "": return
        moduleName = LT( self.editLbl.text()[:-4] ) 
        info = QMessageBox( self )
        info.setText( LT( modulefiles[moduleName].info ) )
        info.setIcon(QMessageBox.Information)
        info.setWindowTitle("Info: " +  LT(moduleName) )
        info.show()


    #############################################################################
    #    04.09.00 Choose File Parameter                                         #
    #############################################################################
    def openFileParam( self ):

        #---Ask For File Name---
        i = self.sender().id
        f = QFileDialog.getOpenFileName(self, ( LT("Choose file") ), path )
        if f == "" or f == ("",""): return  
        #if sys.platform == "win32": 
        #    self.pFile[i] = f[0]
        #elif sys.platform in ["linux2","linux"]: 
        #    self.pFile[i] = f
        self.pFile[i] = f[0]

        self.paramFile[i].setText( os.path.basename(self.pFile[i]) ) 
        app.preview()


    #############################################################################
    #    04.10.00 Add Axis Cross To Draw Preview                                #
    #############################################################################
    def aCross( self ):
         
         #---create axis cross for draw peview---
         xCross = QPainterPath()
         yCross = QPainterPath()
         yCross.moveTo( uni.settings[1]-2, 0 ) 
         yCross.lineTo( uni.settings[1]-2, 130 ) 
         yCross.lineTo( uni.settings[1]-12, 130 ) 
         yCross.lineTo( uni.settings[1], 165 ) 
         yCross.lineTo( uni.settings[1]+12, 130 ) 
         yCross.lineTo( uni.settings[1]+2, 130 ) 
         yCross.lineTo( uni.settings[1]+2, 0 ) 
         yCross.lineTo( uni.settings[1]-2, 0 ) 
         
         xCross.moveTo( uni.settings[1], -2 ) 
         xCross.lineTo( uni.settings[1]-130, -2 ) 
         xCross.lineTo( uni.settings[1]-130, -12 ) 
         xCross.lineTo( uni.settings[1]-165, 0 ) 
         xCross.lineTo( uni.settings[1]-130, +12 ) 
         xCross.lineTo( uni.settings[1]-130, +2 ) 
         xCross.lineTo( uni.settings[1], +2 ) 
         xCross.lineTo( uni.settings[1], -2 ) 
         
         return (xCross, yCross) 


    #############################################################################
    #    04.11.00 Confiurate The Draw Preview                                   #
    #############################################################################
    def paintConfig( self ):
        global win
        
        #---configurate Draw Preview---
        self.paintView.resetTransform()
        self.paintView.rotate(uni.settings[5])
        self.paintView.fitInView( -200,-200,uni.settings[1]+400,
                                  uni.settings[2]+400, Qt.KeepAspectRatio )
        win.viewScale = 1
        

    #############################################################################
    #    04.12.00 Show Module Status                                             #
    #############################################################################
    def moduleStatus( self ): 
        global win
        
        #---Show Module Progress If Available------
        if uni.moduleStatus != "":
            win.statusBar().showMessage(uni.moduleStatus)
        try:
            if (self.mStatusOld != "") and (uni.moduleStatus == ""):
                win.statusBar().showMessage("")
        except: pass
        self.mStatusOld = uni.moduleStatus
        

    #############################################################################
    #    04.13.00 Zoom In                                                       #
    #############################################################################
    def zoomIn( self ): 
        global win

        #---Scale With Buttons---
        factor,d = 1.1,1.0
        win.viewScale *= factor 
        self.paintView.scale(factor, factor)
        app.preview( noCodeUpdate=1 )   
        

    #############################################################################
    #    04.14.00 Zoom Out                                                      #
    #############################################################################
    def zoomOut( self ): 
        global win

        #---Scale With Buttons---
        factor,d = 1.1,1.0
        win.viewScale /= factor 
        self.paintView.scale(1/factor, 1/factor)
        app.preview( noCodeUpdate=1 )   
        

    #############################################################################
    #    04.50.00 Show Error                                                    #
    #############################################################################
    def showError( self, text ):
        self.newBox = QMessageBox( text=text)
        self.newBox.setIcon(QMessageBox.Warning)
        self.newBox.setWindowTitle( "Error" )
        self.newBox.setModal(1)
        self.newBox.show()
        answer = mGui.newBox.exec_()



#################################################################################
#                                                                               # 
#    05.00.00 Joblist Settings                                                  #
#                                                                               # 
#################################################################################
class setGui( QWidget ):
    '''
    #Create a Settings Window. 
    '''
       
    #############################################################################
    #    05.01.00 Init Application Class                                        #
    #############################################################################
    def __init__( self ):
        #---Setup QtWidgets--- 
        QWidget.__init__(self, None, Qt.WindowStaysOnTopHint)
        self.layout()


    #############################################################################
    #    05.02.00 Create Basic Layout                                           #
    #############################################################################
    def layout( self ):
        global path
        global langfiles
        global uni
        global win
        

        #########################################################################
        #    05.02.01 Window Setup                                              #
        #########################################################################

        #---Window Setup---
        self.setGeometry( win.x()+100, win.y()+100, 440, 425 )
        self.setFixedSize( 440, 465 )
        self.setWindowTitle( LT("Lonnox CUT and joblist settings") )
        self.setWindowIcon( QIcon(path + "data/icon.png") )
        

        #########################################################################
        #    05.02.02 Create Layout                                             #
        #########################################################################

        #---Create Setting Widgets---- 
        self.lang = self.combo( LT("Language\n(Restart)"), "language.png", 
                                uni.langList, uni.settings[0], 0, 4) 
        self.xmaxSpin = self.spin( LT("X maximum"), "xmax.png", 
                                   uni.settings[1], 110, 4) 
        self.ymaxSpin = self.spin( LT("Y maximum"), "ymax.png", 
                                   uni.settings[2], 220, 4) 
        self.zmaxSpin = self.spin( LT("Z maximum"), "zmax.png", 
                                   uni.settings[3], 330, 4) 
        self.zSaveSpin = self.spin( LT("Z save height"), "zsave.png", 
                                    uni.settings[4], 0, 220) 
        self.rotSpin = self.spin( LT("Preview\nrotation"), "rotate.png", 
                                  uni.settings[5], 110, 220) 
        self.plungeSpin = self.spin( LT("Plunge feed\nin %"), "plunge.png", 
                                  uni.settings[6], 220, 220) 
        self.bitSpin = self.spin( LT("Holeline\nBit amount"), "drillBit.png", 
                                  uni.settings[7], 330, 220) 
         
        #---Row Separator---  
        r1Sep = QLabel( self )
        r1Sep.setFrameStyle(QFrame.HLine | QFrame.Raised)
        r1Sep.resize(440,4)
        r1Sep.move(0,195)

        #---Row Separator---  
        finSep = QLabel( self )
        finSep.setFrameStyle(QFrame.HLine | QFrame.Raised)
        finSep.resize(440,4)
        finSep.move(0,412)

        #---Create Finish Button---
        finBtn = QPushButton( LT("Finish"), self)
        finBtn.pressed.connect(app.saveSettings)
        finBtn.resize(440-10,25)
        finBtn.move(5,435)
        
            
    #############################################################################
    #    05.03.00 Create Spin Setting                                           #
    #############################################################################
    def spin( self, label, picture, value, x, y ): 

        spnPic = QLabel( self )
        spnPic.setPixmap( QPixmap( path + "data/" + picture ) )
        spnPic.move(x+4,y+2)

        spnLbl = QLabel( label, self )
        spnLbl.setFont( QFont( "Arial",10) )
        spnLbl.setAlignment(Qt.AlignCenter)
        spnLbl.resize(100,60)
        spnLbl.move(x+4,y+102)

        spnVal = QDoubleSpinBox( self )
        spnVal.setMinimum( -1000000.0 )
        spnVal.setMaximum( 1000000.0 )
        spnVal.setValue( value )
        spnVal.resize(100,25)
        spnVal.move(x+4,y+162)

        spnSepR = QLabel( self )
        spnSepR.setFrameStyle(QFrame.VLine | QFrame.Raised)
        spnSepR.resize(4,167)
        spnSepR.move(x+104,y+0)

        spnSepL = QLabel( self )
        spnSepL.setFrameStyle(QFrame.VLine | QFrame.Raised)
        spnSepL.resize(4,167)
        spnSepL.move(x,y+0)

        return spnVal 


    #############################################################################
    #    05.04.00 Create Combo Setting                                          #
    #############################################################################
    def combo( self, label, picture, options, oIndex, x, y ): 
        oIndex = int(oIndex)
        
        cboPic = QLabel( self )
        cboPic.setPixmap( QPixmap( path + "data/" + picture ) )
        cboPic.move(x+4,y+2)

        cboLbl = QLabel( label, self )
        cboLbl.setFont( QFont( "Arial",10) )
        cboLbl.setAlignment(Qt.AlignCenter)
        cboLbl.resize(100,60)
        cboLbl.move(x+4,y+102)

        cboVal = QComboBox( self )
        cboVal.addItems(options)
        cboVal.setCurrentIndex(oIndex)
        cboVal.setFont( QFont( "Arial",10) )
        cboVal.resize(100,25)
        cboVal.move(x+4,y+162)

        cboSepR = QLabel( self )
        cboSepR.setFrameStyle(QFrame.VLine | QFrame.Raised)
        cboSepR.resize(4,167)
        cboSepR.move(x+104,y+0)

        cboSepL = QLabel( self )
        cboSepL.setFrameStyle(QFrame.VLine | QFrame.Raised)
        cboSepL.resize(4,167)
        cboSepL.move(x,y+0)

        return cboVal 


        
#################################################################################
#                                                                               # 
#    06.00.00 Lonnox CUT Application                                            #
#                                                                               # 
#################################################################################
class lcut( object ):
    '''
    #Programm functions are listet here. 
    '''
       
    #############################################################################
    #    06.01.00 Init Application Class                                        #
    #############################################################################
    def __init__( self ):
        global modulefiles
        global langfiles
        global mGui, win
        global uni
        global machine, machines

        #---Create HomePath Folder---
        if not os.path.exists( homePath ): os.makedirs( homePath )
        if not os.path.exists( homePath + "LonnoxCNC" ):
            os.makedirs( homePath + "LonnoxCNC" )
            shutil.copy2( path + "/tools.csv",homePath + "LonnoxCNC" )

        #---Find Machine Configuration Folders---
        machines = [f for f in os.listdir(homePath) if os.path.exists( homePath + f + "/tools.csv" )]
        #print("maschines",machines)
        
        #---Load Last Settings---
        self.read_ini()


    #############################################################################
    #    06.02.00 Load A Module From Module/Job Tree To Edit Area               #
    #############################################################################
    def editJob( self ):
        self.editModule( 0, True)
        
    def editModule( self, event=0, fromJoblist=False ):
        global mGui
        global pEvent
        global joblist
        global machine

        #---Disable Preview Event---
        pEvent = 0
        

        #########################################################################
        #    06.02.01 Reset Edit Area                                           #
        #########################################################################

        #---Reset Edit Section---
        mGui.editLbl.setText("")
        mGui.toolLbl.hide()
        for i in range( len(mGui.paramLbl) ):
            mGui.paramLbl[i].setText("")
            mGui.paramText[i].setText("")
            mGui.paramSpin[i].setText("0.00")
            mGui.paramCombo[i].clear()
            mGui.paramFile[i].setText(""); mGui.pFile[i] = ""
            mGui.paramCheck[i].setCheckState(False)
            mGui.paramMode[i].setCurrentIndex(0)
            mGui.paramMode[i].setEnabled(False)
            mGui.paramMode[i].setStyleSheet("QStackedWidget {}")
        mGui.infoBtn.setEnabled(False)
        mGui.addBtn.setEnabled(False)
        

        #########################################################################
        #    06.02.02 Load Modulename And Set Picture                           #
        #########################################################################

        #---Try To Find The Selected Item In The Module Files---
        try: 
            #---Get Selected Item From The Module or Mask Tree---
            if fromJoblist:
                jobname = LT(mGui.jobTree.currentItem().text(0),True)
                module = jobname[:-4]
                jobNames = [job[1] for job in joblist]
                job = joblist[ jobNames.index(jobname) ]
            else: module = LT( mGui.moduleTree.currentItem().text(0) ) 
            ref = modulefiles[module]
        except (KeyError, ValueError, AttributeError):
            mGui.moduleView.setPixmap(QPixmap())
            pEvent = 1
            self.preview()
            return
        
        #---Check If Job Is Compatible With Avalable Modules---
        if fromJoblist:
            if len(job[2:]) != len(ref.labels):
                mGui.moduleView.setPixmap(QPixmap())
                pEvent = 1
                self.preview()
                return
         
        #---Set Picture If Exist---
        if os.path.exists(  homePath + machine + "/"+ ref.picture ):
            mGui.moduleView.setPixmap( QPixmap( homePath + machine + "/"+ ref.picture ) )
        elif os.path.exists( path + "module/" + ref.picture ):
            mGui.moduleView.setPixmap( QPixmap( path + "module/" + ref.picture ) )
        elif os.path.exists( path + "module Excluded/" + ref.picture ):
            mGui.moduleView.setPixmap( QPixmap( path + "module Excluded/" + ref.picture ) )
        

        #########################################################################
        #    06.02.03 Load Modulelabels And Options                             #
        #########################################################################

        #---Set Labels, And Parameter Widgets---
        for index, label in enumerate(ref.labels):

            #---Set Config Labels And Spinbox And Try To Translate---
            mGui.paramLbl[index].setText( LT(label) )
            if ref.predefinedTools: mGui.toolLbl.show()
        
            #---Set Parameter Widget---
            w = ("E","T","N","O","F","C") #Empty,Text,Number,Options,File,CheckBock
            mGui.paramMode[index].setCurrentIndex( w.index(ref.widget[index]) ) 
            mGui.paramMode[index].setEnabled(True)

            #---Set Options For ComboBox (try translation)---
            if ref.widget[index] == "O":
                if label == "Tool":
                    for no,text,on,off,m6,g43,dia,cdir,typ,ico in uni.toollist:
                        s = no + " - " + text + " " + LT(typ)
                        mGui.paramCombo[index].addItem( QIcon(path+"data/"+ico), s )
                else: mGui.paramCombo[index].addItems( [LT(opt) for opt in ref.options[label]] )


        #########################################################################
        #    06.02.04 Load Parameter From Autofill Table                        #
        #########################################################################
        if not fromJoblist:
            '''
            for index, label in enumerate(ref.labels):
                #---Try To Set Last Known Value With The Autofill List---
                try:
                    i = autofill[0].index( label )
                    if   ref.widget[index] == "T": mGui.paramText[index].setText( autofill[1][i] )
                    elif ref.widget[index] == "N": mGui.paramSpin[index].setText( autofill[1][i] ) 
                    elif ref.widget[index] == "O":
                        oi = ref.options[label].index( autofill[1][i] )
                        mGui.paramCombo[index].setCurrentIndex( oi ) 
                    elif ref.widget[index] == "F":
                        mGui.pFile[index] = autofill[1][i]  
                        mGui.paramFile[index].setText( os.path.basename(mGui.pFile[index]) )
                    elif ref.widget[index] == "C":
                        if autofill[1][i]: mGui.paramCheck[index].setCheckState(True)
                        else: mGui.paramCheck[index].setCheckState(False)
                except ValueError: pass
            '''    
            mGui.addBtn.setText( LT("Add job to joblist -->") )
            mGui.editLbl.setText( LT(module) + " xxx" )
            

        #########################################################################
        #    06.02.05 Load Parameter From Joblist                               #
        #########################################################################
        else: 
            for index, label in enumerate(ref.labels):
                if   ref.widget[index] == "T": mGui.paramText[index].setText(job[index+2])
                elif ref.widget[index] == "N": mGui.paramSpin[index].setText(job[index+2])
                elif ref.widget[index] == "O": 
                    try: oi = ref.options[label].index( job[index+2] )
                    except ValueError: oi = 0
                    mGui.paramCombo[index].setCurrentIndex( oi )    
                elif ref.widget[index] == "F":
                    mGui.pFile[index] = job[index+2]
                    mGui.paramFile[index].setText( os.path.basename(mGui.pFile[index]) )
                elif ref.widget[index] == "C":
                    if job[index+2]: mGui.paramCheck[index].setCheckState(True)
                    else: mGui.paramCheck[index].setCheckState(False)
            mGui.addBtn.setText( LT("Apply parameter -->") )
            mGui.editLbl.setText( LT(job[1],True) )

        #---Validate Formulas In Parameter Fields---
        joblist = self.numberWdgCalc( joblist, validationOnlyMode = 1 ) 
        self.showJobs()
        
        #########################################################################
        #    06.02.06 Finish Moduleloading                                      #
        #########################################################################
        mGui.infoBtn.setEnabled(True)  
        mGui.addBtn.setEnabled(True)  
        pEvent = 1
        mGui.paramSpin[0].setFocus()
        self.preview()



    #############################################################################
    #    06.03.00 Add Job                                                       #
    #############################################################################
    def addJob( self, preview = 0 ):
        global autofill
        global mGui
        global joblist

        #---Extract First Column From Joblist and Modulename---
        jobNames = [job[1] for job in joblist]
        moduleName = LT( mGui.editLbl.text()[:-4] )
        joblistTemp = deepcopy(joblist)
        
        #---Add Parameters To The Job---
        ref = modulefiles[ moduleName ]
        job = []
        for i in range( len(ref.labels) ):
            label = ref.labels[i]

            #---Add Parameter To The Job---
            if   ref.widget[i] == "T": param = mGui.paramText[i].text()
            elif ref.widget[i] == "N": param = mGui.paramSpin[i].text()
            elif ref.widget[i] == "O": 
                options = ref.options[label]
                param = options[ mGui.paramCombo[i].currentIndex() ]
            elif ref.widget[i] == "F": param = mGui.pFile[i]
            elif ref.widget[i] == "C":
                param = mGui.paramCheck[i].checkState()
            job.append( param )
            
            #---Insert Toolinfo For Usage On Different Machines---
            if moduleName=="Toolchange" and i==2 and mGui.paramText[i+1].text()=="":
                s = machine + " " + uni.toollist[int(param)][1] + " " + LT(uni.toollist[int(param)][8])
                param = mGui.paramText[i+1].setText(s)
            #---Update Autofill List (If Module Is Not Toolchange)---
            if moduleName != "Toolchange":
                if label in autofill[0]:
                    autofill[1][ autofill[0].index(label) ] = param
                else: 
                    autofill[0].append( label )
                    autofill[1].append( param )
                
        #---Add New/Edit Job---
        if mGui.editLbl.text()[-1:] == "x": 
            #---Search Next Available Job No---
            if preview: job.insert( 0, moduleName+" xxx") 
            else:
                no = 1
                while "{} {:03d}".format( moduleName, no) in jobNames: no+=1
                job.insert( 0, "{} {:03d}".format( moduleName, no) )
            #---Try To Add Job After The Selected Item Of The Job Tree---
            try:
                item = mGui.jobTree.currentItem().text(0)
                ji = jobNames.index( LT(item,True) ) +1 
            except (AttributeError, ValueError): ji = len( jobNames )
            #---Add JobSetup And Job To Joblist---
            if ref.groupEnd != "":
                joblist.insert( ji, [[0,+1,0]] + job )
                #---If Group Is Added, Add "Group-End" Too---  
                no = 1
                while "{} {:03d}".format(ref.groupEnd,no) in jobNames: no+=1
                joblist.insert( ji+1, [[0,-1,0],"{} {:03d}".format(ref.groupEnd,no),"0.0","0.0"] )
            else: joblist.insert( ji, [[0,0,0]] + job )
        else: 
            ji = jobNames.index( moduleName + mGui.editLbl.text()[-4:] )
            joblist[ji] = job = joblist[ji][:2] + job
        
        #---Save For Preview Purpose---
        if preview: 
            #---Validate Formula In Parameter Fields---
            joblist = self.numberWdgCalc( joblist, validationOnlyMode = 1 ) 
            #---Save Joblist For Preview Purpose---
            joblistP = deepcopy(joblist)
            joblist = deepcopy(joblistTemp) 
            return joblistP    

        #---Update Joblist And Selection---
        self.showJobs()
        self.selectJobs( [ joblist[ji][1] ] )
        
        #---Clear Edit Frame By "doubleclick" On First Child---  
        mGui.moduleTree.setCurrentItem( mGui.moduleTree.topLevelItem(0) )
        self.editModule( )

        #---Update Preview---
        mGui.moduleTree.setFocus()
        self.preview( )

        

    #############################################################################
    #    06.04.00 Show Jobs                                                     #
    #############################################################################
    def showJobs( self ):
        global win
        global mGui
        global joblist
        global modulefiles
        
        #---Validate Formula In Parameter Fields (And Mark Errors)---
        joblist = self.numberWdgCalc( joblist, validationOnlyMode = 1 ) 

        #---Get Selected Job---
        selected = [LT(item.text(0),True) for item in mGui.jobTree.selectedItems()]
        
        #---Delete Old Jobs In The Tree---
        mGui.jobTree.clear()
        mGui.partInfoLbl.setText("")
        
        #---Add Normal And Mirrored Jobs to Job Tree---
        items = []
        groups = []
        for i,job in enumerate(joblist):
            #---Insert Module In Job Tree---
            if len(groups) > 0:
                items.append( QTreeWidgetItem( groups[-1] ) )
            else: items.append( QTreeWidgetItem( mGui.jobTree ) )
            items[-1].setText( 0, LT(job[1],True) ) 

            #---Set Job Color---
            if job[0][0] == 1: brush = QBrush(Qt.gray)
            else: brush = QBrush(Qt.black)
            if sum(job[0][2]) >= 1: 
                items[-1].setExpanded(True)
                items[-1].setForeground( 0,QBrush(Qt.red) )            
            else: items[-1].setForeground( 0,brush )
            
            #---Group Items---
            if job[0][1] == +1: groups.append(items[-1]); mGui.jobTree.expandItem( items[-1] ) 
            elif job[0][1] == -1 and len(groups) > 0: groups.pop()

            #---Insert The Label And Values As Item In The Tree(if moudle available)---
            if not job[1][:-4] in modulefiles: continue 
            ref = modulefiles[ job[1][:-4] ]
            
            #---Check If Jobs Are Compatible With Availabel Jobs/Machine---
            if len(job[2:]) != len(ref.labels):
                paramItem = QTreeWidgetItem( items[-1] )
                paramItem.setText( 0,LT("Incompatibel Modul/Machine") )
                paramItem.setForeground(0,QBrush(Qt.red))
                items[-1].setExpanded(True)
                continue
            
            #---Add Tree Items---
            for j, label in enumerate( ref.labels ):
                if job[0][1] == 0:
                    if label == "": continue
                    if ref.widget[j] == "O":
                        s = "{} : {}".format( LT(label), LT(job[j+2]) ) 
                    else: s = "{} : {}".format( LT(label), str(job[j+2]) ) 
                    paramItem = QTreeWidgetItem( items[-1] )
                    paramItem.setText( 0,s )
                    if job[0][2][j]: paramItem.setForeground( 0,QBrush(Qt.red) )
                    else: paramItem.setForeground( 0,brush ) 
                    
                #---Set Option Icons---
                if label == "Zero Point X": 
                    if (self.isFloat(job[2]) and float(job[2]) != 0) or not self.isFloat(job[2]):
                        items[-1].setIcon(1, QIcon(path + "data/icon_moveX.png")  )
                elif label == "Zero Point Y":
                    if (self.isFloat(job[3]) and float(job[3]) != 0) or not self.isFloat(job[3]):
                        items[-1].setIcon(2, QIcon(path + "data/icon_moveY.png")  )
                elif (label == "Orientation") and (job[4] == "Mirror"):
                    items[-1].setIcon(3, QIcon(path + "data/icon_mirror.png")  )
             
            #---Add Part Information---
            if job[1] == "Part info 001":
                mGui.partInfoLbl.setText( LT(job[1],True) + ":   " + " | ".join(job[5:]) )

        #---Recover Selection---   
        self.selectJobs( selected )

        #---Mark Filename In The Window Title As Unsafed---
        win.setWindowTitle( "*" + appname + " - " + jobfile + "*" )



    #############################################################################
    #    06.04.01 Select Job                                                    #
    #############################################################################
    # selectNames must have joblist language
    def selectJobs( self, selectNames ):
        global mGui
        global joblist

        #---Create JobName List---
        jobNames = [job[1] for job in joblist]
        
        #---Collect Job Tree Items---
        items = []
        for i in range( mGui.jobTree.topLevelItemCount() ):
            #---Append All Top Level Items---
            items.append( mGui.jobTree.topLevelItem(i) )
            #---Append Childs That Are Have Jobnames---
            for j in range( items[-1].childCount() ):
                child = mGui.jobTree.topLevelItem(i).child(j)
                if LT(child.text(0),True) in jobNames:
                    items.append( child )
        
        #---Select Jobs---        
        mGui.jobTree.clearSelection()
        for item in items:        
            if LT(item.text(0),True) in selectNames:
                #---Single--- 
                if len(selectNames) == 1:
                    mGui.jobTree.setCurrentItem( item )
                    break
                #---Multiple--- 
                if len(selectNames) > 1:
                    item.setSelected(True)
                    
    

    #############################################################################
    #    06.05.00 Delete Job                                                    #
    #############################################################################
    def jobDelete( self ):
        global mGui
        global joblist
        global jobManage
        
        #---Repeat For Every Selected Job---
        for item in mGui.jobTree.selectedItems():

            #---Extract First Column From Joblist---
            jobNames = [job[1] for job in joblist]

            #---If Deleted Module Is In Edit Area, Reset It---
            if mGui.editLbl.text() == item.text(0):
                mGui.moduleTree.setCurrentItem( mGui.moduleTree.topLevelItem(0) )
                self.editModule( )
            
            #---Delete Selected Job---
            try: index = jobNames.index( LT(item.text(0),True) ) 
            except (AttributeError,ValueError): return
            del joblist[ index ]
        
        #---Refresh Job Tree---
        self.showJobs()

        #---Select New Job---
        self.selectJobs( [joblist[index-1][1]] )

        #---Update Preview--- 
        self.preview( )


    #############################################################################
    #    06.06.00 Move Job Up                                                   #
    #############################################################################
    def jobUp( self ):
        global mGui
        global joblist
        
        #---Compare Every Job Beginning With The Last---
        newJobs = []
        for index,job in enumerate(joblist):
            
            #---Search Job In Selections---
            for item in mGui.jobTree.selectedItems():
                if LT(item.text(0),True) == job[1]:
                    
                    #---Limit List Index To Available Rows--- 
                    if index <= 0: return

                    #---Move Selected Job---
                    joblist.insert( index-1, joblist.pop( index ) )
                    newJobs.append( deepcopy(joblist[index-1]) )

        #---Select New Job---
        self.showJobs()
        self.selectJobs( [job[1] for job in newJobs ] )

        #---Update Preview---
        self.preview( )


    #############################################################################
    #    06.07.00 Move Job Down                                                 #
    #############################################################################
    def jobDown( self ):
        global mGui
        global joblist


        #---Compare Every Job Beginning With The Last---
        newJobs = []
        for index,job in enumerate(reversed(joblist)):
            index = len(joblist)-1 - index
            
            #---Search Job In Selections---
            for item in mGui.jobTree.selectedItems():
                if LT(item.text(0),True) == job[1]:
                    
                    #---Limit List Index To Available Rows--- 
                    if index >= len(joblist)-1: return

                    #---Move Selected Job---
                    joblist.insert( index+1, joblist.pop( index ) )
                    newJobs.append( deepcopy(joblist[index+1]) )

        #---Select New Job---
        self.showJobs()
        self.selectJobs( [job[1] for job in newJobs ] )

        #---Update Preview---
        self.preview( )


    #############################################################################
    #    06.08.00 Dublicate Jobs                                                #
    #############################################################################
    def jobDuplicate( self ):
        global mGui
        global joblist
        
        #---Extract First Column From Joblist And Get Selected Item---
        jobNames = [job[1] for job in joblist]
        
        #---Repeat For Every Selected Job---
        newJobs = []
        for item in mGui.jobTree.selectedItems():

            #---Find Index Of The Selected Item---
            try:  index = jobNames.index( LT(item.text(0),True) )     
            except: continue      

            #---Search for Next Available Job No---
            i = 1
            jobName = jobNames[index][:-4]
            while True:
                try: jobNames.index( "{} {:03d}".format(jobNames[index][:-4], i) ); i+=1
                except ValueError: jobName = "{} {:03d}".format(jobName, i); break

            #---Insert Job At The End Of The List---        
            newJob = deepcopy(joblist[index])
            newJob[1] = jobName
            newJobs.append( newJob )
            jobNames.append( jobName )
            joblist.append( newJob )
        
        #---Select New Job---
        self.showJobs()
        self.selectJobs( [job[1] for job in newJobs] )

        #---Update Preview---
        self.preview( )


    #############################################################################
    #    06.09.00 Mirror Jobs                                                   #
    #############################################################################
    def jobMirror( self ):
        global mGui
        global joblist
        global modulefiles
        
        #---Show Input Dialog---
        value, ok = QInputDialog.getItem( mGui," ",LT("Jobs"),[LT("Normal"),LT("Mirror")] )
        if not ok: return

        #---Extract First Column From Joblist And Get Selected Item---
        jobNames = [job[1] for job in joblist]
        
        #---Repeat For Every Selected Job---
        selection = []
        for item in mGui.jobTree.selectedItems():

            #---Find Index Of The Selected Item---
            try: index = jobNames.index( LT(item.text(0),True) )     
            except: continue      

            #---Mirror Job If Possible---
            try:
                ref = modulefiles[ jobNames[index][:-4] ].options
                if not "Orientation" in ref: continue 
                if ("Normal" in ref["Orientation"]) and ("Mirror" in ref["Orientation"]):
                    #---Set Orientation Value---
                    joblist[index][4] = LT(value)
                 
                    #---Insert Job At The End Of The List---        
                    selection.append( jobNames[index] )
            except KeyError: continue
                
        #---Select New Job---
        self.showJobs()
        self.selectJobs( selection )
 
        #---Update Preview---
        self.preview( )
 


    #############################################################################
    #    06.11.00 Move Jobs                                                     #
    #############################################################################
    def jobMoveX( self ):
        self.jobMove( "x", LT("Move selected jobs on x axis") )
        
    def jobMoveY( self ):
        self.jobMove( "y", LT("Move selected jobs on y axis") )

    def jobMove( self, xy, txt ):
        global mGui
        global joblist
        global modulefiles

        #---Show Input Dialog---
        value, ok = QInputDialog.getDouble( mGui," ",txt,0.00,-10000,10000,2 )
        if not ok: return
        
        #---Extract First Column From Joblist And Get Selected Item---
        jobNames = [job[1] for job in joblist]
        
        #---Repeat For Every Selected Job---
        selectedJobs = []
        for item in mGui.jobTree.selectedItems():

            #---Find Index Of The Selected Item---
            try:  index = jobNames.index( LT(item.text(0),True) )     
            except: continue      

            #---Move Job If Possible---
            try:
                labels = modulefiles[ jobNames[index][:-4] ].labels
                if (xy == "x") and (labels[0] == "Zero Point X"): 
                    joblist[index][2] = str( float(joblist[index][2]) + value )
                if (xy == "y") and (labels[1] == "Zero Point Y"): 
                    joblist[index][3] = str( float(joblist[index][3]) + value )
                selectedJobs.append( LT(item.text(0),True) )
            except KeyError: continue
            
        #---Select New Job---
        self.showJobs()
        self.selectJobs( selectedJobs )
 
        #---Update Preview---
        self.preview( )



    #############################################################################
    #    06.12.00 Enable/Disable Jobs                                           #
    #############################################################################
    def jobDisable( self ):
        global mGui
        global joblist
        global modulefiles

        #---Show Input Dialog---
        value, ok = QInputDialog.getItem( mGui," ",LT("Jobs"),[LT("Enable"),LT("Disable")] )
        if not ok: return

        
        #---Extract First Column From Joblist And Get Selected Item---
        jobNames = [job[1] for job in joblist]
        
        #---Repeat For Every Selected Job---
        selectedJobs = []
        for item in mGui.jobTree.selectedItems():
            
            #---Check If The Module Is Available---
            try: ref = modulefiles[ LT(item.text(0)[:-4]) ].module              
            except KeyError: 
                mGui.showError( LT("Incompatibel Modul/Machine") )
                continue

            #---Find Index Of The Selected Item---
            try:  
                index = jobNames.index( LT(item.text(0),True) )     
                #---Show/Hide Job If Possible---
                joblist[index][0][0] = float( value == LT("Disable") ) 
                selectedJobs.append( LT(item.text(0),True) )
            except IndexError: continue      
            
        #---Select New Job---
        self.showJobs()
        self.selectJobs( selectedJobs )
 
        #---Update Preview---
        self.preview( )



    #############################################################################
    #    06.40.00 save job settings                                             #
    #############################################################################
    def saveSettings( self ):
        global sGui, win, app, langfiles, mGui
        
        #---Save Job Settings---
        lang = uni.settings[0]
        uni.settings[0] = sGui.lang.currentIndex() 
        uni.settings[1] = sGui.xmaxSpin.value() 
        uni.settings[2] = sGui.ymaxSpin.value() 
        uni.settings[3] = sGui.zmaxSpin.value() 
        uni.settings[4] = sGui.zSaveSpin.value() 
        uni.settings[5] = sGui.rotSpin.value()
        uni.settings[6] = sGui.plungeSpin.value()
        uni.settings[7] = sGui.bitSpin.value()
        
        #---Close Settings Window---
        sGui.close()

        #---Mark Filename In The Window Title As Unsafed---
        win.setWindowTitle( "*" + appname + " - " + jobfile + "*" )

        #---Save Settings In INI File---
        self.write_ini()

        #---Restart Lonnox CUT--- 
        if lang != uni.settings[0]:
            self.loadLanguage()
            win = mainWindow()
            win.show()
            self.showJobs()

            
        #---Update Preview Section---
        mGui.paintConfig() 
        self.preview( )

        

    #############################################################################
    #    06.41.00 File Menu - New                                               #
    #############################################################################
    def menuNew( self ):
        global joblist 
        global jobfile
        global mGui
        global win
        global uni
        
        #---Ask For Saving Changes If Any--- 
        answer = self.askSaveChanges()    
        if answer == 0 or answer == 1:
            #---Clear Edit Frame By "doubleclick" On First Child---  
            mGui.moduleTree.setCurrentItem( mGui.moduleTree.topLevelItem(0) )
            self.editModule( )
            #---Reset Joblist And Cache---
            joblist[:] = []
            self.showJobs()
            uni.cache = {}
            
            #---Remove Filename In The Window Title---
            jobfile = ""
            win.setWindowTitle( appname )

            #---Update Preview---
            self.preview( )


    #############################################################################
    #    06.42.00 File Menu - Open                                              #
    #############################################################################
    def menuOpen( self, event=False, filePath="" ):
        global jobdir , importDir, importFile
        global jobfile
        global joblist
        global mGui
        global win
        global homePath
        global modulefiles

        #---Ask For Saving Changes If Any--- 
        answer = self.askSaveChanges()    

        #---Show File Dialog If Neccessary---
        if filePath == "":
            #---If A Win/Linux Switch Occurs Check The Saved Jobdir---
            if not os.path.exists(jobdir): jobdir = homePath.replace("LonnoxCUT/","")

            #---Get Filename With Dialog---
            file = QFileDialog.getOpenFileName(mGui, (LT("Open file") ),
                                               jobdir, ("Lonnox CUT (*.cut)"))
            #---Adapt Windows PyQt5/ Linux PyQt4---
            if file == "" or file == ("",""): return  
            #if sys.platform == "win32": 
            file = file[0]

            jobfile = file
            jobdir = os.path.dirname(file)
            importDir = ""
            importFile = ""
        else: jobfile = filePath
        
        #---Find The Right Codec---  
        codec = ""
        for c in ["utf-8","cp1252",""]:
            try: 
                with open( jobfile, encoding=c) as file: 
                    data = file.read().split("\n")
                    codec = c 
                    break
            except: pass
        if codec == "": 
            mGui.showError("File encoding error, please contact support.") 
            return
        
        #---Read Data From File If Dialog Wasn't Aborded---
        with open( jobfile, encoding=codec ) as file:
            data = file.read().split("\n")

            #---Show Filename In The Window Title---
            win.setWindowTitle( "*" + appname + " - " + jobfile + "*" )
        
            #---Load Global Variables For Write Access And Delete All Elements---
            joblist[:] = []

            #---Unflatten From String Into 1D/2D Lists---
            section = ""
            ver = 0 
            groupJobs = False
            for row in data:
                #---Skip Empty Rows(TrunCad)---
                if row.strip() == "": continue
                #---Filter Section---
                if row == "[version]": section = "v"
                elif row == "[joblist]": section = "j"
                elif row == "[options]": section = "o"
                #---Save Version---
                elif section == "v" : ver = float(row.replace(".",""))
                #---Add Row To Joblist---
                elif section == "j":
                    if ver < 30: print("Files from Lonnox CUT version 2.9 and earlier are not supported"); break
                    rdata = row.split("\t")
                    job = []
                    for x in rdata:
                        if   x[:1] == "b": job.append( bool(int(x[1:])) )    
                        elif x[:1] == "i": job.append( int(x[1:]) )     
                        elif x[:1] == "f": job.append( x[1:] ) #Convert Floats From Earlier Versions To String    
                        elif x[:1] == "s": job.append( x[1:] )
                        elif x[:1] == "l": 
                            jobSetup = [float(f) for f in x[1:].split(";")]
                            jobSetup.append( [0]*18 ) 
                            job.append(jobSetup)
                    joblist.append( job )
                #---Set Options---
                elif section == "o":
                    if row == "group jobs by tool = True": groupJobs = True                
                    
            #---Reassign Job Numbers If Incompatible (TrunCad)---
            self.jobsNoReassign()
            
            #---Check Modules In Joblist---
            for i in range(len(joblist)):
                if joblist[i][1][:-4] in modulefiles: joblist[i][0][0] = False
                else:
                    joblist[i][0][0] = True
            
            #---Convert Old Group End Modules---
            for i in range(len(joblist)):
                if joblist[i][1].startswith("Contour-End"):
                    while len(joblist[i]) < 4: joblist[i].append(0.0)
 
            #---Convert Old Ellipse Modules---
            for i in range(len(joblist)):
                if ver < 4242 and joblist[i][1].startswith("Ellipse"):
                    del joblist[i][12]              #Last Segment
                    joblist[i].insert(9,"0.00")     #Rotation
                    joblist[i].insert(10,"0.00")    #Segment start
                    joblist[i].insert(11,"360.00")  #Segment end

            #---Group Jobs---
            if groupJobs: self.groupJobs() 
            
            #---Update Joblist And Preview---
            self.showJobs()
            mGui.paintConfig()
            self.preview( )
            self.write_ini()


    #############################################################################
    #    06.43.00 File Menu - Save                                              #
    #############################################################################
    def menuSave( self ):
        global jobdir, jobfile, importFile
        global mGui
        global win
        global homePath

        #---Check Edit Area---
        if mGui.editLbl.text() != "": self.addJob()
        #---If A Win/Linux Switch Occurs Check The Saved Jobdir---
        if not os.path.exists(jobdir): jobdir = homePath.replace("LonnoxCUT/","")
        
        #---Get Filename With Dialog---
        if not os.path.isfile(jobfile):
            if importFile == "": defaultPath = jobdir +"/" + LT("New")
            else: defaultPath = importFile
            fDialog = QFileDialog()
            fDialog.setDefaultSuffix("*.cut")
            file = fDialog.getSaveFileName(mGui, (LT("Save")),
                                           defaultPath + ".cut", ("Lonnox CUT (*.cut)"))
            #---Adapt Windows PyQt5/ Linux PyQt4---
            if file == "" or file == ("",""): return  
            #if sys.platform == "win32": 
            file = file[0]

            jobfile = file
            jobdir = os.path.dirname(file)
            
        #---If File Exist, Save The Data With utf8 Decoding---
        with open( jobfile, encoding="utf8", mode="w" ) as file: 
        
            #---Flatten 2D Joblist Into The Data String---
            data="[version]\n" + __version__ + "\n"
            data += "[joblist]\n" 
            for row in joblist:
                for x in row:
                    if   isinstance(x,bool):  data += "b" + str(int(x))
                    elif isinstance(x,int):   data += "i" + str(x)
                    elif isinstance(x,float): data += "f" + str(x)
                    elif isinstance(x,str):   data += "s" + x
                    elif isinstance(x,list):  data += "l" + ";".join( [str(float(f)) for f in x[:2]] )
                    data += "\t" 
                data += "\n" 
        
            #---Write Data To File If Dialog Wasn't Aborded---
            try:
                file.write( data[:-1] )
                file.close
                #---Show Filename In The Window Title---
                win.setWindowTitle( "" + appname + " - " + jobfile + "" )
                #---Save Filename For Next Save Action---
                self.write_ini()
            except: pass
        


    #############################################################################
    #    06.44.00 File Menu - Save As                                           #
    #############################################################################
    def menuSaveAs( self ):
        global jobfile
        
        jobfile = ""
        self.menuSave()
        


    #############################################################################
    #    06.45.00 File Menu - Save Gcode                                        #
    #############################################################################
    def menuSaveGcode( self ):
        global gcodedir, importFile
        global jobfile
        global mGui
        global homePath

        #---Check Edit Area---
        if mGui.editLbl.text() != "": self.addJob()

        #---Check Area---
        self.generateGCode()

        #---If A Win/Linux Switch Occurs Check The Saved Jobdir---
        if not os.path.exists(gcodedir): gcodedir = homePath.replace("LonnoxCUT/","")
        
        #---Ask For Filename Of The Gcode File---
        if importFile == "": defaultPath = gcodedir + "/" + os.path.basename(jobfile).split(".")[0]
        else: defaultPath = importFile
        fDialog = QFileDialog()
        fDialog.setDefaultSuffix("*.ngc")
        saveFile = QFileDialog.getSaveFileName(mGui, (LT("Save gcode file")),
                                               defaultPath + ".ngc",
                                               ("LinuxCNC (*.ngc)"))
        #---Adapt Windows PyQt5/ Linux PyQt4---
        if saveFile == "" or saveFile == ("",""): return  
        #if sys.platform == "win32": 
        saveFile = saveFile[0]
        
        gcodedir = os.path.dirname(saveFile)
        
        #---If File Is Choosen Then Open It With utf8 Encoding---
        file = open(saveFile, mode="w", encoding="utf-8") 

        #---Build GCode And Write To File---
        file.write( self.code )
        file.close
        self.write_ini()

        

    #############################################################################
    #    06.46.00 File Menu - Programm beenden                                  #
    #############################################################################
    def menuExit( self ):

        self.write_ini()
        sys.exit()



    #############################################################################
    #    06.47.00 Option Menu - Add New Modules                                 #
    #############################################################################
    def menuAddModules( self ):
        global app, win, homePath, machines

        #---Get Filename With Dialog---
        file = QFileDialog.getOpenFileName(mGui, (LT("Open file") ),
                                           homePath.replace("LonnoxCUT/",""), (" (*.zip)"))
        #---Adapt Windows PyQt5/ Linux PyQt4---
        if file == "" or file == ("",""): return  
        #if sys.platform == "win32": 
        file = file[0]
        
        if not os.path.exists(homePath): os.makedirs( homePath)
        
        #---Zip File Extraction--- 
        zip_ref = zipfile.ZipFile(file, 'r')
        zip_ref.extractall(homePath)
        zip_ref.close() 

        #---Load Modules--- 
        self.loadModules()

        #---Find Machine Configuration Folders---
        machines = [f for f in os.listdir(homePath) if os.path.exists( homePath + f + "/tools.csv" )]

        #---Restart Lonnox CUT---
        win = mainWindow()
        win.show()
        self.showJobs()



    #############################################################################
    #    06.49.00 Menu - Ask For Save Changes                                   #
    #############################################################################
    def askSaveChanges( self ):
        global win,mGui
        
        #---Ask For Saving If Title Contains Asterix---  
        answer = 0
        if win.windowTitle().startswith("*"):
            #---Show "Jobs Will Be Deleted" Message---
            mGui.newBox = QMessageBox( text=LT("Save changes?") )
            mGui.newBox.addButton(LT("No"), QMessageBox.NoRole)
            mGui.newBox.addButton(LT("Yes"), QMessageBox.YesRole)
            mGui.newBox.addButton(LT("Abort"), QMessageBox.RejectRole)
            mGui.newBox.setIcon(QMessageBox.Question)
            mGui.newBox.setWindowTitle( " " )
            mGui.newBox.setModal(1)
            mGui.newBox.show()
            answer = mGui.newBox.exec_()
        
        #---Delete Joblist---
        if answer == 1:
            self.menuSave()

        return answer    

        
        
    #############################################################################
    #    06.80.00 Write ini File                                                #
    #############################################################################
    def write_ini( self ):
        global win
        global jobdir, importDir, gcodedir
        global homePath
        global machine
        
        #---LonnoxCUT Settings: Flatten Data To String---
        data  = ("version  :" + __version__          + "\n" +
                 "wxpos    :" + str( win.x() )       + "\n" +
                 "wypos    :" + str( win.y() )       + "\n" +
                 "jobdir   :" + jobdir               + "\n" +
                 "gcodedir :" + gcodedir             + "\n" +
                 "importDir:" + importDir            + "\n" +
                 "language :" + str(uni.settings[0]) + "\n" +
                 "machine  :" + machine              + "\n" +
                 "g0Mode   :" + str(uni.g0Mode)      + "\n")
                 
        #---LonnoxCut Settings: Write Data---
        with open( homePath + "LSetV30.ini", encoding="utf8", mode="w" ) as file: 
            file.write(data)
         
        #---Machine Settings: Flatten Data To String---
        data  = ("version  :" + __version__          + "\n" +
                 "xMax     :" + str(uni.settings[1]) + "\n" +
                 "yMax     :" + str(uni.settings[2]) + "\n" +
                 "zMax     :" + str(uni.settings[3]) + "\n" +
                 "zSave    :" + str(uni.settings[4]) + "\n" +
                 "drawRot  :" + str(uni.settings[5]) + "\n" +
                 "plunge   :" + str(uni.settings[6]) + "\n" +
                 "bit      :" + str(uni.settings[7]) + "\n" +
                 "m5Wait   :" + str(uni.m5Wait) + "\n" )
                 
        #---Machine Settings: Write Data---
        with open( homePath + machine +"/MSetV30.ini", encoding="utf8", mode="w" ) as file: 
            file.write(data)

        

    #############################################################################
    #    06.81.00 Read ini File                                                 #
    #############################################################################
    def read_ini( self, skipMachine=False):
        global wxpos, wypos
        global jobdir, gcodedir, importDir
        global appname
        global langfiles
        global homePath
        global machine

        #---LonnoxCUT Settings: Read Data From File---
        if os.path.exists( homePath + "LSetV30.ini" ): 
            with open( homePath + "LSetV30.ini", encoding="utf-8", mode="r" ) as file:
                data = file.read().split("\n")                     
                section = ""
                ver = 0 
                for row in data:
                    #---Filter Section---
                    if   row[:10] == "version  :": ver = row[10:]
                    elif row[:10] == "wxpos    :": wxpos = int(row[10:])
                    elif row[:10] == "wypos    :": wypos = int(row[10:])
                    elif row[:10] == "jobdir   :": jobdir = row[10:]
                    elif row[:10] == "gcodedir :": gcodedir = row[10:]
                    elif row[:10] == "importDir:": importDir = row[10:]
                    elif row[:10] == "language :": uni.settings[0] = float(row[10:])
                    elif row[:10] == "machine  :" and not skipMachine: machine = row[10:]
                    elif row[:10] == "g0Mode   :": uni.g0Mode = int(row[10:])
                    
        #---Check Machine Availability---
        if machine not in machines: machine = machines[0]        

        #---Machine Settings: Read Data From File---
        if os.path.exists( homePath + machine + "/MSetV30.ini" ): 
            with open( homePath + machine + "/MSetV30.ini", encoding="utf-8", mode="r" ) as file:
                data = file.read().split("\n")    
                for row in data:
                    #---Filter Section---
                    if   row[:10] == "xMax     :": uni.settings[1] = float(row[10:])
                    elif row[:10] == "yMax     :": uni.settings[2] = float(row[10:])
                    elif row[:10] == "zMax     :": uni.settings[3] = float(row[10:])
                    elif row[:10] == "zSave    :": uni.settings[4] = float(row[10:])
                    elif row[:10] == "drawRot  :": uni.settings[5] = float(row[10:])
                    elif row[:10] == "plunge   :": uni.settings[6] = float(row[10:])
                    elif row[:10] == "bit      :": uni.settings[7] = float(row[10:])
                    elif row[:10] == "m5Wait   :": uni.m5Wait = float(row[10:])

        

    #############################################################################
    #    06.82.00 GenerateGCode                                                 #
    #############################################################################
    def generateGCode(  self, insertEditJob=0, pMode=0 ):
        global win
        global joblist
        global jobfile
        global modulefiles
        global mGui
        global debugMode
        
        #---Update module Code If module Available--- 
        if (mGui.editLbl.text() != "") and insertEditJob: joblistP = self.addJob( 1 )
        else: joblistP = deepcopy(joblist)

        #---Delete Deactivatet Jobs---
        joblistP = [job for job in joblistP if not job[0][0]]
        
        #---Calculate Formulas In Parameter Fields---
        joblistP = self.numberWdgCalc( joblistP ) 

        #---Collect Tool Data---
        jTools = []
        for index, job in enumerate( joblistP ):
            try: 
                ref = modulefiles[ job[1][:-4] ].module 
                t = ref.tool( app, joblistP, index )
                tNew = [t[0],t[1],[],[]] #[tStaNr,tEndNr,tStaValues,tEndValues]
                #---If Tool Is A List, Append Additional Values In A Compatible Format--- 
                if isinstance(t[0],list): tNew[0] = t[0][0]; tNew[2] = t[0][1:] 
                if isinstance(t[1],list): tNew[1] = t[1][0]; tNew[3] = t[1][1:] 
                jTools.append( tNew )
            except KeyError: pass 
        
        #---Collect Gcode Data---
        uni.oNumber = 100
        uni.oStack = []
        uni.moduleStatus = ""
        code = ""
        for index, job in enumerate( joblistP ):
            if debugMode:
                ref = modulefiles[ job[1][:-4] ]
                code += ref.module.gcode( app, jTools, joblistP, index, pMode ) 
            else:
                try:
                    ref = modulefiles[ job[1][:-4] ]
                    code += ref.module.gcode( app, jTools, joblistP, index, pMode )  
                except (KeyError,IndexError): pass    
            
        #---And Now All Together ;-) ---
        self.code  = ("%\n(" + ("-"*30) + appname + ("-"*30) + ")\n" +
                      "(" + LT("File") + ": " + os.path.basename(jobfile) + ")\n\n" +
                      code + "\n\n" + "M2\n")

        return


    #############################################################################
    #    06.82.01 Number (Spinbox) Widget Check                                 #
    #############################################################################
    # The python comand "eval" will execute a string as if it were written as 
    # programm line. To prevent damage through inputs like "os.remove(...)" only
    # digits are forwarded to eval.
    def numberWdgCalc( self, joblistTemp, validationOnlyMode = 0 ):
        global modulefiles, mGui, joblist

        #---Dont Change Original---
        jobsTemp = deepcopy(joblistTemp)
        
        #---Reset Error Marks Of Parameter Fields---
        for i in range(20):
            mGui.paramMode[i].setStyleSheet("QStackedWidget { }")

        #---Itterate Through Joblist---
        #[ print("2277 wdgcalc",job) for job in joblistTemp ]
        vars = []
        for i,job in enumerate(jobsTemp):
            #---Get Module Reference---
            moduleName = job[1][:-4]  
            try: ref = modulefiles[ moduleName ]
            except KeyError: continue
            #---Check If Jobs Are Compatible With Availabel Jobs/Machine---
            if len(job[2:]) != len(ref.labels): 
                jobsTemp[i][0][0] = True
                continue
            #---Reset Error Marks---
            jobsTemp[i][0][2] = [0]*20
            #---Extract Variable Contents---
            if moduleName == "Variable":
                #---Check Variable Name---
                if self.isFloat(job[4]): 
                    jobsTemp[i][0][2][2] = 1
                    if LT(mGui.editLbl.text(),True) == job[1]:
                        mGui.paramMode[2].setStyleSheet("QStackedWidget { border: 3px solid #c00; }")
                #---Check Variable Value---
                if not self.isFloat(job[5]):
                    jobsTemp[i][0][2][3] = 1
                    if LT(mGui.editLbl.text(),True) == job[1]:
                        mGui.paramMode[3].setStyleSheet("QStackedWidget { border: 3px solid #c00; }")
                #---Add Variable If No Errors And Module Is Visible--- 
                if sum(jobsTemp[i][0][2][2:4]) == 0 and not job[0][0]:
                    vars.insert( 0,job[4:6] )
                continue    
            #---Itterate Through Parameter Of The Job---
            for j,parameter in enumerate(job[2:]): 
                #---Calculate Number Widget Content---
                if ref.widget[j] == "N":
                    #---Split Math Formula String---
                    p = parameter.replace(" ","").replace(",",".")
                    formula = []
                    start = 0
                    for k,c in enumerate(p):                    
                        if c in ["+","-","*","/"]:
                            formula.append(p[start:k])
                            formula.append(p[k])
                            start = k+1
                    formula.append(p[start:])        
                    #---Replace Vars With Value---
                    for k,f in enumerate(formula):
                        for (vn,vv) in vars:
                            if f == vn: formula[k] = vv; break
                    #---Check For Zero Division Error---
                    for l,f in enumerate(formula):
                        if l > 0 and self.isFloat(f) and float(f) == 0.00 and formula[l-1] == "/":
                            formula[l] = "1.00"
                    #---Check For Bad Sign Combination---
                    fCheck = "".join(formula)
                    if fCheck != "":
                        for c1 in ["+","-","*","/"]:
                            for c2 in ["+","-","*","/"]:
                                if c1+c2 in fCheck: fCheck = "bad sign combination"
                        if fCheck[0] in ["*","/"] or fCheck[-1] in ["*","/"]: fCheck = "bad sign combination"         
                        if fCheck.count(".") > 1: fCheck = "bad sign combination"
                    #---Format String For Validation---
                    for c in ["+","-","*","/","."]:
                        fCheck = fCheck.replace(c,"")
                    #---Execute Eval If String Contains Only Digits (See Method Notes)---
                    result = None
                    if fCheck.isdigit(): 
                        try: result = eval( "".join(formula) )
                        except: pass
                    #---Apply Result---
                    if result != None:
                        if not validationOnlyMode: jobsTemp[i][j+2] = result
                    #---Or Mark Errors---
                    else: 
                        if not validationOnlyMode: jobsTemp[i][j+2] = 0.0
                        jobsTemp[i][0][2][j] = 1
                        if LT(mGui.editLbl.text(),True) == job[1]:
                            mGui.paramMode[j].setStyleSheet("QStackedWidget { border: 3px solid #c00; }")
        return jobsTemp  


    #############################################################################
    #    06.82.02 Check Float
    #############################################################################
    def isFloat( self, string ):
        try:
            float(string)
            return True
        except:
            return False


    #############################################################################
    #    06.83.00 Update View                                                   #
    #############################################################################
    def preview(  self, event=0, noCodeUpdate=0 ):
        global mGui
        global win
        global joblist
        global modulefiles
        global pEvent
        global uni


        #########################################################################
        #    06.83.01 Build GCode                                               #
        #########################################################################
        
        #---Draw If Necessary---
        if not pEvent: return 
        
        #---Preview Mode---
        if mGui.viewMode.currentIndex() == 2: mode = 0
        else: mode = 1

        #---Build GCode---
        if noCodeUpdate == 0:
            self.generateGCode( 1, mode )
            mGui.codeView.setText( self.code )
            self.jPath = self.drawGCode( uni.settings[1], 0 )
 

        #########################################################################
        #    06.83.02 Draw GCode                                                #
        #########################################################################

        #---Reset Code Preview And StausBar---
        mGui.scene.clear()
        wa = QPainterPath()
        wa.addRect(0,0,uni.settings[1],uni.settings[2])
        mGui.scene.addPath(wa, QPen(Qt.black, 3, Qt.DotLine))
        mGui.scene.setSceneRect( -1000,-200,uni.settings[1]+2000,uni.settings[2]+400 )
        
        #---Scale Text--- 
        textPath = QPainterPath()
        for text in self.drawText:
            if text[0]: continue
            tSize = int(40/win.viewScale +1)
            pt = QPainterPath()
            pt.addText( 0, 0, QFont( "Arial",tSize), text[1] )
            mx = QTransform()
            mx.translate( text[2], text[3] )
            mx.rotate(-uni.settings[5])
            textPath.addPath( mx.map(pt) )

        #---Add Job Drawings If jPath Already Exist--- 
        try:
            axis = mGui.aCross()
            mGui.scene.addPath(self.jPath["raw"], brush=QBrush(Qt.darkMagenta) )           
            mGui.scene.addPath( axis[0], QPen(Qt.transparent), QBrush(Qt.red)) 
            mGui.scene.addPath( axis[1], QPen(Qt.transparent), QBrush(Qt.blue)) 
            modulePen = QPen(Qt.white, 0, Qt.SolidLine, Qt.RoundCap, Qt.RoundJoin) 
            mGui.scene.addPath( textPath , QPen(Qt.transparent), QBrush(QColor("#c0c0c0")) )              
            mGui.scene.addPath(self.jPath["cut"], modulePen)                                   
            if uni.g0Mode: mGui.scene.addPath(self.jPath["g0"], QPen(Qt.darkCyan, 2, Qt.DashLine))              
        except (TypeError, AttributeError, UnboundLocalError): pass
        
        

    #############################################################################
    #    06.84.00 Draw GCode                                                    #
    #############################################################################
    def drawGCode( self, x, y ):
        global win
        
        #---Setup Cenversion Values---
        gLines = self.code.split("\n")
        lmax = len(gLines)
        c = [x,y,0,0,0,0,0]                   
        cutPath = QPainterPath()
        g0Path = QPainterPath()
        rawPath = QPainterPath()
        self.drawText = [] #PositionValid,text,xPix,yPix]
        xyPlane = 1
        
        #---Convert G Commands To Painter Path---
        win.key = ""
        for index, line in enumerate(gLines):
            #---Show Progress In StatusBar---
            win.statusBar().showMessage(LT("Calculating preview (ESC): ") + str(index)+"/"+str(lmax)) 
            #---Cancel Drawing With ESC---
            QCoreApplication.processEvents()
            if win.key == "ESC": win.key = ""; break
            #---Draw G0 Lines---
            if   line[:3] == "G0 ":
                start = list(c)
                gcode.gExtract( lcut, line, c, uni.settings[1] )
                g0Path.moveTo( *start[:2] )
                g0Path.lineTo( *c[:2] )
            #---Draw G1 Lines---
            elif line[:3] == "G1 ":
                start = list(c)
                gcode.gExtract( lcut, line, c, uni.settings[1] )
                cutPath.moveTo( *start[:2] )
                cutPath.lineTo( *c[:2] )
                #---Draw Immerse Drill---
                if ("Z" in line) and uni.g0Mode:
                    cutPath.moveTo(c[0]-5,c[1]-5)
                    cutPath.lineTo(c[0]+5,c[1]+5)
                    cutPath.moveTo(c[0]-5,c[1]+5)
                    cutPath.lineTo(c[0]+5,c[1]-5)
                #---Try To Draw Modulename---
                try:
                    if self.drawText[-1][0]:                
                        self.drawText[-1] = [ 0, self.drawText[-1][1], c[0], c[1] ] 
                except (NameError,ValueError): pass
            #---Draw G2 And G3 Circles---
            elif line[:3] == "G2 " or line[:3] == "G3 ":
                start = list(c)
                gcode.gExtract( lcut, line, c, uni.settings[1] )
                qCoords = gcode.g23Convert(lcut, line[:2], start[0], start[1], *c[:5] )
                if xyPlane:
                    cutPath.arcMoveTo( *qCoords[:-1] )
                    cutPath.arcTo( *qCoords )
            #---Draw G5 Cubic Bezier---
            elif line[:3] == "G5 ":
                start = list(c)
                gcode.gExtract( lcut, line, c, uni.settings[1] )
                cutPath.moveTo( *start[:2] )
                cutPath.cubicTo( start[0]+c[2],start[1]+c[3],c[0]+c[5],c[1]+c[6],c[0],c[1] )
            #---Draw G5 Quadratic Bezier---
            elif line[:5] == "G5.1 ":
                start = list(c)
                gcode.gExtract( lcut, line, c, uni.settings[1] )
                cutPath.moveTo( *start[:2] )
                cutPath.quadTo( start[0]+c[2],start[1]+c[3],c[0],c[1] )
            #---Disable Arcs On Planes Other Then XY---
            elif line[:4] in ["G18 ","G19 "] or line[:6] in ["G17.1","G18.1","G19.1"]:
                xyPlane = 0
            #---Enable Arcs On XY Planes---
            elif line[:4] == "G17 ":
                xyPlane = 1
            #---Draw Rawpart---
            elif line[:4] == "(" + LT("Rawpart")[:3]:
                qCoords = gcode.rExtract( lcut, line, uni.settings[1] )
                rawPath.addRect( *qCoords )
            #---Draw Modulename---
            elif line[:4] == "(---":
                self.drawText.append( [1,line[31:line.find("---",31)],0,0] )
            #---Contour Point---
            elif (line[:3] == "(." + LT("Point")[:1]) or (line[:3] == "(." + LT("Spline-Point")[:1]):
                cp = line.split()
                self.drawText.append( [0,cp[0][1:],uni.settings[1]-float(cp[1][1:]),float(cp[2][1:-1])] )
        #---Status Bar zurücksetzen---
        win.statusBar().showMessage("")
        
        return {"cut":cutPath, "g0":g0Path, "raw":rawPath, "x":c[0], "y":c[1]}



    #############################################################################
    #    06.85.00 Load Modules                                                  #
    #############################################################################
    def loadModules( self ):
        global modulefiles, mGui, win, machine, machines, dxfFilter
        
        #---Search Module Files And Add New Import Directory---
        sys.path.insert(0, path + "module/")
        files = os.listdir(path+"module/")
        if os.path.exists(path+"module Excluded/"):
            sys.path.insert(0, path+"module Excluded/")
            files = os.listdir(path+"module Excluded/") + files
        if os.path.exists(homePath + "/" + machine):
            sys.path.insert(0, homePath + machine)
            files = os.listdir(homePath + machine) + files
        
        #---Find *.py Files In Search Result And Import Them---
        modulefiles = {}
        for file in files:
            fBase = file.split(".")[0]
            #---Only Import First Occurrence Of A File---
            if (not fBase in modulefiles) and file.endswith(".py"):
                #---Import Or Reload Modules---
                if not fBase in sys.modules: __import__( fBase )
                else: reload( sys.modules[ fBase ] )
                ref = sys.modules[ fBase ]
                #---Save Module Name And Object Reference In A Dictionary---
                if fBase == "dxfFilter":
                    dxfFilter = ref
                else:
                    modulefiles[ref.name] = ref
         


    #############################################################################
    #    06.86.00 Load Language                                                 #
    #############################################################################
    def loadLanguage( self ):
        global machine, machines, win, L, langfiles
        
        #---Load Language---
        langKey = uni.langList[int(uni.settings[0])]        
        L = langfiles[langKey].language

        #---Add Items For Opposite translation to L---
        L2 = L.copy()
        for i,j in L2.items():
            if not j in L2: L[j] = i

        #---Add Items For Info Message---
        L.update( langfiles[langKey].infos )

        #---Update uni.language---
        uni.language = L


    #############################################################################
    #    06.87.00 New Machines                                                  #
    #############################################################################
    def newMachine( self ):
        global machine, machines, win, modulefiles, mGui, joblist
        
        #---Set Current Machine---
        if win != None : 
            machine = machines[ win.machineCombo.currentIndex() ] 
            self.read_ini( skipMachine=True ) #lädt neue einstellungen und alte maschine
            self.write_ini() #schreibt neue einstellungen und neue maschine
            self.createToollist()
            self.loadModules()        
             
            #---Reload Module Tree---
            section = {} 
            mGui.moduleTree.clear() 
            for name, ref in sorted(modulefiles.items()):
                if not ref.section in section:
                    section[ref.section] = QTreeWidgetItem(mGui.moduleTree)
                    section[ref.section].setText(0, LT(ref.section) )
                QTreeWidgetItem(section[ref.section]).setText(0, LT(ref.name) )

            #---Hide Jobs That Are Not Available In Modules---
            for i in range(len(joblist)):
                moduleName = joblist[i][1][:-4]  
                if moduleName in modulefiles: 
                    joblist[i][0][0] = False
                    #---Check If Jobs In List And Modules Have The Same Amount Of Parameters---
                    if len(joblist[i][2:]) != len(modulefiles[moduleName].labels): 
                        joblist[i][0][0] = True
                else: joblist[i][0][0] = True
            
            #---Clear Edit Frame By "doubleclick" On First Child---  
            mGui.moduleTree.setCurrentItem( mGui.moduleTree.topLevelItem(0) )
            self.editModule( 0 )
            self.showJobs()
            
        

    #############################################################################
    #    06.88.00 Create Toollist                                               #
    #############################################################################
    def createToollist( self ):
        global homePath, machine
        
        #---Load Toolnames From File---
        uni.toollist = []
        with open( homePath + machine + "/tools.csv", newline="", encoding="utf8") as csvfile:
            uni.toollist = [row for i,row in enumerate( csv.reader(csvfile)) if i>0 ]



    #############################################################################
    #    06.89.00 Group Jobs By Tool                                            #
    #############################################################################
    def groupJobs( self ):
        global joblist, mGui
        
        #---Build Tool List---
        t = ""
        jobTools = []  #[tool,applied]
        for job in joblist:
            if job[1].startswith("Toolchange"): t = job[4]
            jobTools.append([t,False])

        #---Itterate Through Jobs, Skip If Already Applied---
        jobs = []
        for iA,jobA in enumerate(joblist): 
            if jobTools[iA][1]: continue
            #---Append First Toolchange---
            if jobA[1].startswith("Toolchange"):
                jobs.append(jobA)
                jobTools[iA][1] = True
                #---Search For Equal Tools---
                for iB,jobB in enumerate(joblist):
                    if jobTools[iB][1]: continue
                    #---Find Jobs With Same Toolnumber--- 
                    if jobTools[iB][0] == jobA[4]:
                        #---Ignore Repeated Equal Toolchanges---
                        if jobB[1].startswith("Toolchange"): jobTools[iB][1] = True
                        #---Add Jobs And Mark As Applied---
                        if not jobTools[iB][1]:
                            jobs.append(jobB)
                            jobTools[iB][1] = True
            #---Append Jobs That Are Not Related To Last Toolchange---
            else:                                                                
                jobs.append(jobA)
                jobTools[iA][1] = True
         
        joblist = jobs   



    #############################################################################
    #    06.90.00 Check Job Numbers                                             #
    #############################################################################
    # The function is necessarry because TrunCad will give job numbers that are
    # more that take more then 4 Bytes, which is not allowed in LCUT. The function
    # will reassign job numbers that are more than 3 Bytes in length.
    def jobsNoReassign( self ):
        global joblist
        
        #---Search For Jobnumber With Overlegth---
        for job in joblist:
            if (len(job[1]) - job[1].rfind(" ")) > 4:            
                #---Reassign All Jobs Numbers---
                for i in range(len(joblist)):
                    noI = joblist[i][1].rfind(" ")
                    joblist[i][1] = joblist[i][1][:noI] + " {:03d}".format(i)                 
                break                
        


    #############################################################################
    #    06.91.00 Menu Import dxf Files                                         #
    #############################################################################
    def menuDxfImport( self ):
        global importDir
        global importFile
        global joblist
        global mGui, win
        global homePath
        global modulefiles

        #---If A Win/Linux Switch Occurs Check The Saved Jobdir---
        if not os.path.exists(importDir): importDir = homePath.replace("LonnoxCUT/","")

        #---Get Filename With Dialog---
        file = QFileDialog.getOpenFileName(mGui, (LT("Import dxf file...") ),
                                           importDir, ("Dxf "+LT("File")+" (*.dxf)"))
        #---Adapt Windows PyQt5/ Linux PyQt4---
        if file == "" or file == ("",""): return  
        #if sys.platform == "win32": 
        file = file[0]
        
        importDir = os.path.dirname(file)
        importFile = file[:-4]
        
        #---Show Filename In The Window Title---
        win.setWindowTitle( "*" + appname + " - " + jobfile + "*" )

        #---Choose Custom Filter--- 
        filter, ok = QInputDialog.getItem( win,LT("Import dxf file..."),LT("Choose a dxf filter:"),
                                           dxfFilter.getNames() )
        if not ok: return

        #---TrunCad Imports Have Seperate Import Method For Historical Reasons---
        if filter == "TrunCad":
            text = LT("X/Y Offset to format stop-edges of workpeace [mm]:")
            offset, ok = QInputDialog.getDouble( mGui," ",text,0.00,-1000,1000,2 )
            joblist += convert.truncad( lcut, joblist, file, offset )
        #---DXF Imports With Standard And Custom Filters---
        else:
            entys = dxf.read(file)
            joblist = dxfFilter.filter( joblist,entys,filter,file )
            #---Group Jobs---
            self.groupJobs() 
            
        #---Check If Importet Modules Are Available---
        for i in range(len(joblist)):
            if not joblist[i][1][:-4] in modulefiles: joblist[i][0][0] = True
        
        #---Update Joblist And Preview---
        self.showJobs()
        mGui.paintConfig()
        self.preview( )



#################################################################################
#                                                                               #
#   07.00.00 Language Translation                                               #
#                                                                               #
#################################################################################
def LT( term, isJobname=0 ):
    
    #---split jobname---
    if isJobname: name = term[:-4]; no = term[-4:]
    else: name = term; no = ""
    
    #---try to translate---  
    try: name = uni.language[name]
    except KeyError: pass
    return name + no


#################################################################################
#                                                                               #
#   07.01.00 App Setup                                                          #
#                                                                               #
#################################################################################
def appSetup( ):
    
    #---Create Toollist---
    app.createToollist()
    
    #---Search Files In ./lang/ And Add New Import Directory---
    sys.path.insert(0, path + "lang/")

    #---Find *.py Files In Search Result And Import Them---
    for file in os.listdir(path+"lang/"):
        if file.endswith(".py"):
            file = file.split(".")[0]
            __import__( file )
            ref = sys.modules[ file ]
            #---Save Module Name And Object Reference In A Dictionary---
            langfiles[file] = ref


    #---Create Languagelist with English first---
    uni.langList = sorted(langfiles)
    uni.langList.insert(0,uni.langList.pop(uni.langList.index("English")))

    #---Load Language---
    app.loadLanguage()
    
    #---Load Modules--- 
    app.loadModules()



#################################################################################
#                                                                               #
#   08.00.00 Run Application                                                    #
#                                                                               #
#################################################################################
def main():
    global app, win, joblist
   
    #---Set Widgets Compatibe With Windows Display Scaling---
    QApplication.setAttribute(Qt.AA_EnableHighDpiScaling, True)
    QApplication.setAttribute(Qt.AA_UseHighDpiPixmaps, True) 

    #---Execute Lonnox CUT Program Functions---
    app = lcut()

    #---Execute/Refresh Qt User Interfase Functions--- 
    qApp = QApplication([])
    
    #---Splash Screen---
    splash = QSplashScreen(QPixmap(path + "data/splash.png"))
    splash.show()
    splash.showMessage("<p><font face='verdana' color='#fff'>Version "+__version__+"</font></p>", 
                       Qt.AlignBottom | Qt.AlignCenter)
    
    #---Load Timeintense Filecontent--- 
    appSetup()   

    win = mainWindow()
    win.show()
    splash.finish(win)

    #---Open File If Available---
    if len(sys.argv) > 1 and sys.argv[1].endswith(".cut"): 
        app.menuOpen( filePath=sys.argv[1] )
    
    sys.exit(qApp.exec_())

#---Run The Main Loop---
if __name__ == '__main__':
    main()

